program SimpleChat;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


uses
  BaseUnix, SysUtils, ChatAI, DateUtils, StrUtils, Math, 
  PluginManagerUnit, WeatherPlugin, MathPlugin, DateTimePlugin,
  PoetryPlugin, SoundPlugin, PluginInterface, Unix;

var
  AI: TChatAI;
  UserInput: string;
  ShouldExit: Boolean = False;
  PlugManager: TPluginManager;

procedure HandleSigInt(sig: cint); cdecl;
begin
  ShouldExit := True;
end;

procedure CheckPlugins;
var
  Plugins: TPluginArray;
  i: Integer;
  TestCases: array of string = (
    'погода',
    'сколько времени',
    '2+2',
    'стихи о любви',
    'звук вкл'
  );
begin
  WriteLn(#13#10'Детальная проверка плагинов:');
  Plugins := PlugManager.GetPlugins;
  
  for i := 0 to High(Plugins) do
  begin
    WriteLn('   ', i+1, '. ', Plugins[i].GetName);
    WriteLn('      Тестовый запрос: "', TestCases[i], '"');
    
    try
      if Plugins[i].CanHandle(TestCases[i]) then
      begin
        Write('      Может обработать: Да, результат: ');
        WriteLn(Plugins[i].HandleInput(TestCases[i]));
      end
      else
        WriteLn('      Может обработать: Нет');
    except
      on E: Exception do
        WriteLn('      Ошибка: ', E.Message);
    end;
  end;
end;

procedure Initialize;
var PoetryP: TPoetryPlugin;
begin
  WriteLn('Инициализация системы...');
  
  // 1. Инициализация ядра
  WriteLn('1. Создаем ядро чат-бота');
  AI := TChatAI.Create;
  
  // 2. Инициализация менеджера плагинов
  WriteLn('2. Инициализируем менеджер плагинов');
  PlugManager := TPluginManager.Create;
  
  // 3. Регистрация плагинов с проверкой
  WriteLn('3. Регистрируем плагины:');
  
  try
    // WeatherPlugin
    Write('   - WeatherPlugin: ');
    PlugManager.RegisterPlugin(TWeatherPlugin.Create);
    WriteLn('OK');
    
    // DateTimePlugin
    Write('   - DateTimePlugin: ');
    PlugManager.RegisterPlugin(TDateTimePlugin.Create);
    WriteLn('OK');
    
    // MathPlugin
    Write('   - MathPlugin: ');
    PlugManager.RegisterPlugin(TMathPlugin.Create);
    WriteLn('OK');
    
  // PoetryPlugin с расширенной проверкой
  Write('   - PoetryPlugin: ');
  PoetryP := TPoetryPlugin.Create;
  try
    if PoetryP.CanHandle('тест стих') then
    begin
      PlugManager.RegisterPlugin(PoetryP);
      WriteLn('OK (', PoetryP.GetName, ')');
      
      // Автозагрузка стихов
      if FileExists('poetry_corpus.txt') then
      begin
        WriteLn('     Загружаем poetry.txt...');
        PoetryP.LoadPoetryCorpus('poetry_corpus.txt');
      end;
    end
    else
    begin
      WriteLn('FAIL (initialization check failed)');
      PoetryP.Free;
    end;
  except
    on E: Exception do
    begin
      WriteLn('FAIL (', E.Message, ')');
      if Assigned(PoetryP) then PoetryP.Free;
    end;
  end;
    
    // SoundPlugin - с особой проверкой
    Write('   - SoundPlugin: ');
    if fpSystem('command -v gorg64_spkplay') = 0 then
    begin
      PlugManager.RegisterPlugin(TSoundPlugin.Create);
      WriteLn('OK (sound player found)');
    end
    else
    begin
      WriteLn('SKIPPED (gorg64_spkplay not found)');
    end;
    
  except
    on E: Exception do
      WriteLn('Ошибка инициализации плагинов: ', E.Message);
  end;
end;

procedure ShowCapabilities;
begin
  WriteLn('Я умею:');
  WriteLn('- Отвечать на вопросы (база знаний)');
  WriteLn('- Показывать текущее время');
  WriteLn('- Решать математические выражения');
  WriteLn('- Показывать погоду (пример: "погода в Москве")');
  WriteLn('- Сочинять стихи (команды: "стихи о любви", "шутка в стихах")');
  WriteLn('- Звуковое сопровождение (команды: "звук вкл", "звук выкл")');
  WriteLn('- Обучаться на новых стихах (команда: "загрузить стихи")');
  WriteLn('- Переобучать модели (команда: "обучить модели")');
  WriteLn;
end;

procedure ProcessUserInput;
var
  LowerInput: string;
  Response: string;
begin
  LowerInput := LowerCase(Trim(UserInput));
  
  // Пропускаем пустые запросы
  if LowerInput = '' then Exit;
  
  // Специальные команды
  if LowerInput = 'помощь' then
  begin
    ShowCapabilities;
    Exit;
  end;
  
  if LowerInput = 'проверить плагины' then
  begin
    CheckPlugins;
    Exit;
  end;
  
  // Обработка через плагины
  Response := PlugManager.HandleInput(UserInput);
  
  // Форматированный вывод
  if Response <> '' then
  begin
    WriteLn(#27'[1;36m', FormatDateTime('hh:nn:ss', Now), #27'[0m ', 
      #27'[1;32mИИ:'#27'[0m ', Response);
  end
  else
  begin
    WriteLn(#27'[1;36m', FormatDateTime('hh:nn:ss', Now), #27'[0m ',
      #27'[1;31mИИ:'#27'[0m Не удалось обработать запрос');
  end;
end;

procedure Finalize;
begin
  WriteLn('Завершение работы...');
  try
    if Assigned(AI) then
      AI.Free;
    if Assigned(PlugManager) then
      PlugManager.Free;
    WriteLn('Чат завершён.');
  except
    on E: Exception do
      WriteLn('Ошибка при завершении: ', E.Message);
  end;
end;

begin
  FpSignal(SIGINT, @HandleSigInt);
  
  Initialize;
  CheckPlugins;
  ShowCapabilities;
  
  try
    while not ShouldExit do
    begin
      Write('Вы: ');
      ReadLn(UserInput);
      
      if ShouldExit then Break
      else if UserInput = 'пока' then Break
      else if Trim(UserInput) <> '' then
      begin
        try
          ProcessUserInput;
        except
          on E: Exception do
            WriteLn(#27'[31m[Ошибка]'#27'[0m ', E.ClassName, ': ', E.Message);
        end;
      end;
    end;
  finally
    Finalize;
  end;
end.