unit WeatherPlugin;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$MODE OBJFPC}{$H+}

interface

uses
  PluginInterface, SysUtils, Process, fpjson, jsonparser, DateUtils, classes, strutils;

type
  TWeatherPlugin = class(TInterfacedObject, IPlugin)
  private
    FLastWeather: string;
    FLastUpdate: TDateTime;
    function TryWttrIn(const City: string; out Weather: string): Boolean;
    function TryOpenWeatherMap(const City: string; out Weather: string): Boolean;
    function GetWeatherFromAPI(const City: string): string;
  public
    function CanHandle(const Input: string): Boolean;
    function HandleInput(const Input: string): string;
    function GetName: string;
  end;

function GetWeatherInfo(const City: string = ''): string;

implementation

const
  OPENWEATHER_API_KEY = 'ваш_api_ключ'; // Замените на ваш ключ!

function TWeatherPlugin.TryWttrIn(const City: string; out Weather: string): Boolean;
var
  Process: TProcess;
  OutputStream: TMemoryStream;
  Buffer: array[1..2048] of byte;
  BytesRead: LongInt;
  JsonString: string;
  JsonData: TJSONData;
begin
  Result := False;
  Process := TProcess.Create(nil);
  OutputStream := TMemoryStream.Create;
  try
    Process.Executable := 'curl';
    Process.Parameters.Add('https://wttr.in/' + City + '?format=j1');
    Process.Parameters.Add('--user-agent');
    Process.Parameters.Add('Mozilla/5.0 (Windows NT 10.0; Win64; x64)');
    Process.Options := [poUsePipes];

    Process.Execute;
    while Process.Running or (Process.Output.NumBytesAvailable > 0) do
    begin
      BytesRead := Process.Output.Read(Buffer, SizeOf(Buffer));
      OutputStream.Write(Buffer, BytesRead);
    end;

    SetLength(JsonString, OutputStream.Size);
    OutputStream.Position := 0;
    OutputStream.ReadBuffer(JsonString[1], OutputStream.Size);

    if (Length(JsonString) > 50) and (Pos('temp_C', JsonString) > 0) then
    begin
      JsonData := GetJSON(JsonString);
      try
        Weather := JsonData.FindPath('current_condition[0].temp_C').AsString + '°C, ' +
                   JsonData.FindPath('current_condition[0].weatherDesc[0].value').AsString;
        Result := True;
        WriteLn('[Плагин погоды] wttr.in успешно ответил');
      finally
        JsonData.Free;
      end;
    end;
  finally
    Process.Free;
    OutputStream.Free;
  end;
end;

function TWeatherPlugin.TryOpenWeatherMap(const City: string; out Weather: string): Boolean;
var
  Process: TProcess;
  OutputStream: TMemoryStream;
  Buffer: array[1..2048] of byte;
  BytesRead: LongInt;
  JsonString: string;
  JsonData: TJSONData;
  CityID, TempStr, Description: string;
begin
  Result := False;
  
  // Определяем ID города
  if Pos('moscow', LowerCase(City)) > 0 then
    CityID := '524901' // Москва
  else if (Pos('saint', LowerCase(City)) > 0) or (Pos('petersburg', LowerCase(City)) > 0) then
    CityID := '498817' // СПб
  else
    CityID := '524901'; // По умолчанию Москва

  Process := TProcess.Create(nil);
  OutputStream := TMemoryStream.Create;
  try
    Process.Executable := 'curl';
    Process.Parameters.Add('https://api.openweathermap.org/data/2.5/weather?id=' + CityID + 
                          '&units=metric&appid=' + OPENWEATHER_API_KEY); // Убрали lang=ru
    Process.Options := [poUsePipes];

    Process.Execute;
    while Process.Running or (Process.Output.NumBytesAvailable > 0) do
    begin
      BytesRead := Process.Output.Read(Buffer, SizeOf(Buffer));
      OutputStream.Write(Buffer, BytesRead);
    end;

    SetLength(JsonString, OutputStream.Size);
    OutputStream.Position := 0;
    OutputStream.ReadBuffer(JsonString[1], OutputStream.Size);

    if (Length(JsonString) > 10) and (Pos('temp', JsonString) > 0) then
    begin
      JsonData := GetJSON(JsonString);
      try
        // Форматируем температуру
        TempStr := FormatFloat('0.0', JsonData.FindPath('main.temp').AsFloat);
        
        // Берём описание на английском (например, "light rain")
        Description := JsonData.FindPath('weather[0].main').AsString;
        
        // Переводим основные варианты на русский
        if Description = 'Rain' then Description := 'дождь'
        else if Description = 'Clouds' then Description := 'облачно'
        else if Description = 'Clear' then Description := 'ясно'
        else if Description = 'Snow' then Description := 'снег';
        
        Weather := TempStr + '°C, ' + Description;
        Result := True;
        WriteLn('[Плагин погоды] OpenWeatherMap успешно ответил');
      finally
        JsonData.Free;
      end;
    end;
  finally
    Process.Free;
    OutputStream.Free;
  end;
end;

function TWeatherPlugin.GetWeatherFromAPI(const City: string): string;
begin
  // Сначала пробуем wttr.in
  if not TryWttrIn(City, Result) then
  begin
    WriteLn('[Плагин погоды] wttr.in не ответил, пробуем OpenWeatherMap...');
    if not TryOpenWeatherMap(City, Result) then
      Result := 'Не удалось получить данные о погоде';
  end;
end;

function TWeatherPlugin.HandleInput(const Input: string): string;
var
  City, WeatherData, DisplayCity: string;
begin
  // Определяем город
  if (Pos('москв', Input) > 0) or (Pos('moscow', LowerCase(Input)) > 0) then
  begin
    City := 'Moscow';
    DisplayCity := 'Москве';
  end
  else if (Pos('санкт-петербург', Input) > 0) or 
          (Pos('петербург', Input) > 0) or 
          (Pos('spb', LowerCase(Input)) > 0) then
  begin
    City := 'Saint+Petersburg';
    DisplayCity := 'Санкт-Петербурге';
  end
  else
  begin
    City := 'Moscow';
    DisplayCity := 'Москве';
  end;

  // Проверяем кэш
  if (MinutesBetween(Now, FLastUpdate) < 30) and (FLastWeather <> '') then
  begin
    WriteLn('[Плагин погоды] Используем кэшированные данные');
    Result := FLastWeather;
    Exit;
  end;

  // Получаем свежие данные
  WeatherData := GetWeatherFromAPI(City);
  
  // Обновляем кэш
  FLastWeather := WeatherData;
  FLastUpdate := Now;
  
  Result := Format('Текущая погода в %s: %s (обновлено %s)', 
    [DisplayCity, WeatherData, FormatDateTime('hh:nn', Now)]);
end;

function TWeatherPlugin.CanHandle(const Input: string): Boolean;
begin
  Result := Pos('погод', LowerCase(Input)) > 0;
end;

function TWeatherPlugin.GetName: string;
begin
  Result := 'WeatherPlugin v2.4 (с резервированием API)';
end;

function GetWeatherInfo(const City: string = ''): string;
var
  Process: TProcess;
  OutputStream: TStream;
  BytesRead: LongInt;
  Buffer: array[1..2048] of byte;
  JsonString: string;
  JsonData: TJSONData;
begin
  Result := 'Не удалось получить данные о погоде';
  
  Process := TProcess.Create(nil);
  try
    Process.Executable := 'curl';
    if City = '' then
      Process.Parameters.Add('https://wttr.in/?format=j1')
    else
      Process.Parameters.Add('https://wttr.in/' + City + '?format=j1');
    
    Process.Options := [poUsePipes];
    Process.Execute;
    
    OutputStream := TMemoryStream.Create;
    try
      while Process.Running or (Process.Output.NumBytesAvailable > 0) do
      begin
        BytesRead := Process.Output.Read(Buffer, SizeOf(Buffer));
        OutputStream.Write(Buffer, BytesRead);
      end;
      
      SetLength(JsonString, OutputStream.Size);
      OutputStream.Position := 0;
      OutputStream.ReadBuffer(JsonString[1], OutputStream.Size);
      
      try
        JsonData := GetJSON(JsonString);
        try
          Result := 'Текущая погода: ' + 
                    JsonData.FindPath('current_condition[0].temp_C').AsString + '°C, ' +
                    JsonData.FindPath('current_condition[0].weatherDesc[0].value').AsString;
        finally
          JsonData.Free;
        end;
      except
        on E: Exception do
          Result := 'Ошибка обработки данных: ' + E.Message;
      end;
    finally
      OutputStream.Free;
    end;
  finally
    Process.Free;
  end;
end;

end.