program word2vectranslator;

{
    Word2Vec Translator - простой переводчик на основе Word2Vec
    Для GNU/Linux 64 bit version.
    Version: 1.0
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025 Artyomov Alexander
    http://self-made-free.ru/
    aralni@mail.ru

    Лицензия: GNU Affero General Public License
}

{$MODE OBJFPC}{$H+}
{$CODEPAGE UTF8}
{$INLINE ON}
{$RANGECHECKS ON}

uses
  SysUtils, Classes, DataUtils, Word2Vec, ucs4unit, ucs4opunit, ucs4functionsunit, 
  ucs4multilineunit, LazUTF8;

type
  TTranslationOptions = record
    SourceModel: string;
    TargetModel: string;
    InputFile: string;
    OutputFile: string;
    TranslationThreshold: Double;
    MaxSimilarWords: Integer;
  end;

var
  Options: TTranslationOptions;

procedure PrintHelp;
begin
  Writeln('Word2Vec Translator - перевод на основе семантических векторов');
  Writeln('Использование:');
  Writeln('  word2vectranslator <source_model> <target_model> <input_file> <output_file>');
  Writeln('');
  Writeln('Параметры:');
  Writeln('  source_model    - модель Word2Vec исходного языка');
  Writeln('  target_model    - модель Word2Vec целевого языка');
  Writeln('  input_file      - файл с текстом для перевода');
  Writeln('  output_file     - файл для сохранения перевода');
  Writeln('');
  Writeln('Дополнительные опции (через переменные окружения):');
  Writeln('  TRANS_THRESHOLD=0.7   - порог схожести для перевода (0.0-1.0)');
  Writeln('  MAX_SIMILAR=3         - количество кандидатов для выбора');
end;

function ParseCommandLine: TTranslationOptions;
begin
  if ParamCount < 4 then
  begin
    PrintHelp;
    Halt(1);
  end;

  Result.SourceModel := ParamStr(1);
  Result.TargetModel := ParamStr(2);
  Result.InputFile := ParamStr(3);
  Result.OutputFile := ParamStr(4);

  // Чтение дополнительных параметров из переменных окружения
//  Result.TranslationThreshold := StrToFloatDef(GetEnvironmentVariable('TRANS_THRESHOLD'), 0.7);
// В ParseCommandLine измените значение по умолчанию
Result.TranslationThreshold := StrToFloatDef(GetEnvironmentVariable('TRANS_THRESHOLD'), 0.85); // Было 0.7

  Result.MaxSimilarWords := StrToIntDef(GetEnvironmentVariable('MAX_SIMILAR'), 3);
end;

function LoadTextFile(const Filename: string): ucs4multiline;
begin
  Result.Init;
  Result.LoadFromFile(Filename);
end;

procedure SaveTextFile(const Filename: string; const Text: ucs4multiline);
begin
  Text.SaveToFile(Filename);
end;

function TranslateWord(SourceEmbeddings, TargetEmbeddings: TWordEmbeddings; 
                      const Word: ucs4; Threshold: Double; MaxCandidates: Integer): ucs4;
var
  LowerWord: ucs4;
  SimilarWords: TStringArray;
  SimilarityScores: array of Double;
  BestScore: Double;
  BestIndex: Integer;
  i: Integer;
  WordStr: string;
begin
  WordStr := Word.ToUTF8;
  Writeln('Перевод слова: "', WordStr, '"');
  
  // Пропускаем короткие слова и знаки препинания
  if (Word.Length <= 2) or IsPunctuation(Word[0]) then
  begin
    Writeln('  Пропускаем (короткое слово или пунктуация)');
    Result := Word;
    Exit;
  end;

  LowerWord := ToLower(Word);
  
  try
    SimilarWords := TargetEmbeddings.MostSimilar(LowerWord.ToUTF8, MaxCandidates * 2); // Берем больше кандидатов
    
    if Length(SimilarWords) = 0 then
    begin
      Writeln('  Нет похожих слов в целевой модели');
      Result := Word;
      Exit;
    end;

    // Фильтруем кандидатов
    SetLength(SimilarityScores, Length(SimilarWords));
    BestScore := 0;
    BestIndex := -1;

    for i := 0 to High(SimilarWords) do
    begin
      SimilarityScores[i] := SourceEmbeddings.FastSimilarity(
        LowerWord.ToUTF8, SimilarWords[i]);
      
      Writeln('  Кандидат "', SimilarWords[i], '": ', SimilarityScores[i]:0:3);
      
      // Дополнительная проверка: отбрасываем слишком короткие или неподходящие слова
      if (SimilarityScores[i] > BestScore) and 
         (Length(SimilarWords[i]) >= 2) and // Минимальная длина
         (SimilarityScores[i] >= Threshold) then
      begin
        BestScore := SimilarityScores[i];
        BestIndex := i;
      end;
    end;

    if (BestIndex >= 0) and (BestScore >= Threshold) then
    begin
      Result := SimilarWords[BestIndex];
      Writeln('  Выбрано: "', Result.ToUTF8, '" (схожесть: ', BestScore:0:3, ')');
    end
    else
    begin
      Writeln('  Подходящий перевод не найден (лучший score: ', BestScore:0:3, ')');
      Result := Word;
    end;

  finally
    LowerWord.Clear;
  end;
end;

function TranslateSentence(SourceEmbeddings, TargetEmbeddings: TWordEmbeddings;
                          const Sentence: ucs4; Threshold: Double; MaxCandidates: Integer): ucs4;
var
  Tokens, TranslatedTokens: TUC4Array;
  i: Integer;
begin
  // Токенизируем предложение
  Tokens := TokenizeForNLP(Sentence);

for i := 0 to High(Tokens) do
WriteLn('Debug:  Tokens[i] = ', Tokens[i].ToUTF8);
  
  if Length(Tokens) = 0 then
  begin
    Result := Sentence;
    Exit;
  end;

  SetLength(TranslatedTokens, Length(Tokens));
  
  // Переводим каждый токен
  for i := 0 to High(Tokens) do
  begin
    // Пропускаем знаки препинания и короткие слова
    if (Tokens[i].Length <= 1) or IsPunctuation(Tokens[i][0]) then
      TranslatedTokens[i] := Tokens[i]
    else
      TranslatedTokens[i] := TranslateWord(SourceEmbeddings, TargetEmbeddings, 
                                         Tokens[i], Threshold, MaxCandidates);
  end;

for i := 0 to High(TranslatedTokens) do
WriteLn('Debug:  TranslatedTokens[i] = ', TranslatedTokens[i].ToUTF8);

  // Собираем обратно в предложение
  Result := Join(TranslatedTokens, $20); // Пробел как разделитель

WriteLn('Debug: (Собираем обратно в предложение) Result = ', Result.ToUtf8);

  // Очищаем память
  for i := 0 to High(Tokens) do
    Tokens[i].Clear;
  for i := 0 to High(TranslatedTokens) do
    if TranslatedTokens[i].FData <> Tokens[i].FData then // Избегаем двойного освобождения
      TranslatedTokens[i].Clear;
end;

procedure TranslateFile(const Options: TTranslationOptions);
var
  SourceEmbeddings, TargetEmbeddings: TWordEmbeddings;
  InputText, OutputText: ucs4multiline;
  TranslatedLine: ucs4;
  i: Integer;
  StartTime, EndTime: TDateTime;
begin
  Writeln('Загрузка исходной модели: ', Options.SourceModel);
  StartTime := Now;
  SourceEmbeddings := TWordEmbeddings.Create(Options.SourceModel);
  Writeln('Исходная модель загружена за ', FormatDateTime('nn:ss.zzz', Now - StartTime));
  Writeln('Словарь: ', SourceEmbeddings.FVocab.Count, ' слов');

  Writeln('Загрузка целевой модели: ', Options.TargetModel);
  StartTime := Now;
  TargetEmbeddings := TWordEmbeddings.Create(Options.TargetModel);
  Writeln('Целевая модель загружена за ', FormatDateTime('nn:ss.zzz', Now - StartTime));
  Writeln('Словарь: ', TargetEmbeddings.FVocab.Count, ' слов');

  Writeln('Загрузка исходного текста: ', Options.InputFile);
  InputText := LoadTextFile(Options.InputFile);
  Writeln('Загружено строк: ', InputText.Count);

  OutputText.Init;
  
  Writeln('Начало перевода...');
  StartTime := Now;
  
  for i := 0 to InputText.Count - 1 do
  begin
    if InputText[i].Length > 0 then
    begin
      TranslatedLine.Init;
      TranslatedLine := TranslateSentence(SourceEmbeddings, TargetEmbeddings, 
                                        InputText[i], Options.TranslationThreshold, 
                                        Options.MaxSimilarWords);
      OutputText.AddCopy(TranslatedLine);
      TranslatedLine.Clear;
    end
    else
    begin
      OutputText.Add(Default(ucs4)); // Пустая строка
    end;
WriteLn('>>> ', OutputText[0].ToUTF8);

    // Прогресс
    if (i + 1) mod 100 = 0 then
      Writeln('Переведено строк: ', i + 1, ' из ', InputText.Count);
  end;

  EndTime := Now;
  Writeln('Перевод завершен за ', FormatDateTime('nn:ss.zzz', EndTime - StartTime));

  Writeln('Сохранение перевода: ', Options.OutputFile);
  SaveTextFile(Options.OutputFile, OutputText);

WriteLn('Очистка памяти');
  // Очистка памяти
//  SourceEmbeddings.Free;
//  TargetEmbeddings.Free;
  InputText.Clear;
  OutputText.Clear;
  
  Writeln('Перевод успешно завершен!');
end;

begin
  try
    Writeln('Word2Vec Translator v1.0');
    Writeln('=======================');
    
    Options := ParseCommandLine;
    
    Writeln('Настройки:');
    Writeln('  Порог схожести: ', Options.TranslationThreshold:0:2);
    Writeln('  Кандидатов: ', Options.MaxSimilarWords);
    Writeln('');

    TranslateFile(Options);
    
  except
    on E: Exception do
    begin
      Writeln('Ошибка: ', E.Message);
      Halt(1);
    end;
  end;
end.