program ruconverter;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$mode objfpc}{$H+}
{$MODESWITCH UNICODESTRINGS}
{$CODEPAGE UTF8}

uses
  SysUtils, classes, rustringunit, Math;

procedure ShowHelp;
begin
  writeln('Использование:');
  writeln('  Конвертация UTF-8 -> RUS: ruconverter <input.utf8> <output.rus>');
  writeln('  Конвертация RUS -> UTF-8: ruconverter -d <input.rus> <output.utf8>');
  writeln;
  writeln('Параметры:');
  writeln('  -d  Режим декодирования (из RUS в UTF-8)');
  halt(1);
end;

// Вместо загрузки всего файла можно читать построчно
procedure ProcessLargeFile(const InputFile, OutputFile: string);
var
  InF, OutF: Text;
  Line: string;
  RusLine: Rustring;
begin
  Assign(InF, InputFile);
  Assign(OutF, OutputFile);
  Reset(InF);
  Rewrite(OutF);
  try
    while not Eof(InF) do
    begin
      ReadLn(InF, Line);
      RusLine := UTF8ToRus(Line);
      // Запись в бинарный формат потребует дополнительной работы
    end;
  finally
    Close(InF);
    Close(OutF);
  end;
end;

procedure ConvertUTF8ToRus(const InputFile, OutputFile: string);
var
  Input: Text;
  Output: TRusFile;
  Line: UTF8String;
  LineCount: LongInt = 0;
begin
  Assign(Input, InputFile);
  Reset(Input);
  try
    // Первый проход - подсчёт строк
    while not EOF(Input) do
    begin
      ReadLn(Input, Line);
      Inc(LineCount);
    end;
    
    // Выделение памяти
    SetLength(Output, LineCount);
    
    // Второй проход - обработка
    Reset(Input);
    LineCount := 0;
    while not EOF(Input) do
    begin
      ReadLn(Input, Line);
      Output[LineCount] := UTF8ToRus(Line);
      Inc(LineCount);
      
      // Прогресс каждые 1000 строк
      if LineCount mod 1000 = 0 then
        WriteLn('Обработано: ', LineCount, ' строк');
    end;
    
    SaveRusFile(OutputFile, Output);
    WriteLn('Успешно конвертировано. Всего строк: ', LineCount);
  finally
    Close(Input);
  end;
end;

procedure ConvertRusToUTF8(const InputFile, OutputFile: string);
var
  Input: TRusFile;
  Output: Text;
  I: LongInt;
begin
  Input := LoadRusFile(InputFile);
  Assign(Output, OutputFile);
  Rewrite(Output);
  try
    for I := 0 to High(Input) do
    begin
      WriteLn(Output, RusToUTF8(Input[I]));
      
      // Прогресс каждые 1000 строк
      if I mod 1000 = 0 then
        WriteLn('Обработано: ', I, ' из ', Length(Input), ' строк');
    end;
    WriteLn('Успешно конвертировано. Всего строк: ', Length(Input));
  finally
    Close(Output);
  end;
end;

var
  DecodeMode: Boolean = False;
  InputFile, OutputFile: string;
begin
  // Парсинг аргументов командной строки
  if (ParamCount < 2) or (ParamCount > 3) then
    ShowHelp;

  if ParamStr(1) = '-d' then
  begin
    if ParamCount <> 3 then
      ShowHelp;
    DecodeMode := True;
    InputFile := ParamStr(2);
    OutputFile := ParamStr(3);
  end
  else
  begin
    if ParamCount <> 2 then
      ShowHelp;
    InputFile := ParamStr(1);
    OutputFile := ParamStr(2);
  end;

  // Проверка существования входного файла
  if not FileExists(InputFile) then
  begin
    writeln('Ошибка: входной файл не существует - ', InputFile);
    halt(3);
  end;

  // Выполнение конвертации
  if DecodeMode then
    ConvertRusToUTF8(InputFile, OutputFile)
  else
    ConvertUTF8ToRus(InputFile, OutputFile);
end.