unit rustringcoreunit;
{$mode objfpc}{$H+}
{$modeswitch advancedrecords}
{$MODESWITCH UNICODESTRINGS}
{$CODEPAGE UTF8}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  Classes, SysUtils;

type
  TRusCharMap = record
    Char: UnicodeChar;
    Code: Byte;
  end;

const
  RusCharMapAi: array of TRusCharMap = (
    // === КРИТИЧНО ВАЖНЫЕ СИМВОЛЫ ДЛЯ ИИ (0x00-0x3F) ===
    
    // Спецсимволы с высокой семантической нагрузкой (0x00-0x0F)
    (Char: ' ';  Code: $00),  // Пробел - самый частый
    (Char: '.';  Code: $01),  // Точка
    (Char: ',';  Code: $02),  // Запятая  
    (Char: '!';  Code: $03),  // Восклицание
    (Char: '?';  Code: $04),  // Вопрос
    (Char: ':';  Code: $05),  // Двоеточие
    (Char: ';';  Code: $06),  // Точка с запятой
    (Char: '''; Code: $07),   // Апостроф
    (Char: '"';  Code: $08),  // Кавычки
    (Char: '-';  Code: $09),  // Дефис
    (Char: '_';  Code: $0A),  // Подчеркивание
    (Char: '/';  Code: $0B),  // Слеш
    (Char: '\';  Code: $0C),  // Обратный слеш
    (Char: '@';  Code: $0D),  // Собака
    (Char: '#';  Code: $0E),  // Хештег
    (Char: '$';  Code: $0F),  // Доллар

    // === БУКВЫ С ФОНЕТИЧЕСКИМ ГРУППИРОВАНИЕМ (0x10-0x3F) ===
    
    // Латинские и русские с схожим звучанием РЯДОМ!
    (Char: 'a'; Code: $10), (Char: 'а'; Code: $11),  // a-а
    (Char: 'b'; Code: $12), (Char: 'б'; Code: $13),  // b-б
    (Char: 'c'; Code: $14), (Char: 'к'; Code: $15),  // c-к (фонетически близко)
    (Char: 'd'; Code: $16), (Char: 'д'; Code: $17),  // d-д
    (Char: 'e'; Code: $18), (Char: 'е'; Code: $19),  // e-е
    (Char: 'f'; Code: $1A), (Char: 'ф'; Code: $1B),  // f-ф
    (Char: 'g'; Code: $1C), (Char: 'г'; Code: $1D),  // g-г
    (Char: 'h'; Code: $1E), (Char: 'х'; Code: $1F),  // h-х
    (Char: 'i'; Code: $20), (Char: 'и'; Code: $21),  // i-и
    (Char: 'j'; Code: $22), (Char: 'ж'; Code: $23),  // j-ж
    (Char: 'k'; Code: $24), (Char: 'к'; Code: $25),  // k-к
    (Char: 'l'; Code: $26), (Char: 'л'; Code: $27),  // l-л
    (Char: 'm'; Code: $28), (Char: 'м'; Code: $29),  // m-м
    (Char: 'n'; Code: $2A), (Char: 'н'; Code: $2B),  // n-н
    (Char: 'o'; Code: $2C), (Char: 'о'; Code: $2D),  // o-о
    (Char: 'p'; Code: $2E), (Char: 'п'; Code: $2F),  // p-п
    (Char: 'q'; Code: $30), (Char: 'к'; Code: $31),  // q-к
    (Char: 'r'; Code: $32), (Char: 'р'; Code: $33),  // r-р
    (Char: 's'; Code: $34), (Char: 'с'; Code: $35),  // s-с
    (Char: 't'; Code: $36), (Char: 'т'; Code: $37),  // t-т
    (Char: 'u'; Code: $38), (Char: 'у'; Code: $39),  // u-у
    (Char: 'v'; Code: $3A), (Char: 'в'; Code: $3B),  // v-в
    (Char: 'w'; Code: $3C), (Char: 'в'; Code: $3D),  // w-в
    (Char: 'x'; Code: $3E), (Char: 'кс'; Code: $3F), // x-кс

    // Продолжение алфавита (0x40-0x5F)
    (Char: 'y'; Code: $40), (Char: 'й'; Code: $41),  // y-й
    (Char: 'z'; Code: $42), (Char: 'з'; Code: $43),  // z-з
    (Char: 'A'; Code: $44), (Char: 'А'; Code: $45),  // A-А
    (Char: 'B'; Code: $46), (Char: 'Б'; Code: $47),  // B-Б
    // ... остальные прописные

    // === ЦИФРЫ И РЕДКИЕ СИМВОЛЫ (0x60-0xFF) ===
    (Char: '0'; Code: $60), (Char: '1'; Code: $61),
    // ... остальные цифры
    (Char: '№'; Code: $FF)  // Номер
  );

function FindCharCode(C: WideChar): Byte;
function FindCharByCode(Code: Byte): WideChar;

implementation

function FindCharCode(C: UnicodeChar): Byte;
var i:LongInt;
begin
  // Быстрая проверка цифр
  if (C >= '0') and (C <= '9') then
    Exit(Ord(C) - Ord('0'));

  // Явная обработка специальных символов
//  case C of
//    ' ': Exit($0A);  // Пробел
//    '.': Exit($0F);  // Точка
//    ',': Exit($A0);  // Запятая - новый код
//    '!': Exit($0B);
//    '?': Exit($0E);
//    ';': Exit($0C);
//    ':': Exit($0D);
    // Остальные специальные символы...
//  end;

  // Поиск в таблице
  for I := 0 to High(RusCharMap) do
    if RusCharMap[I].Char = C then
      Exit(RusCharMap[I].Code);

  // Неизвестный символ
  Result := $FF;
end;

function FindCharByCode(Code: Byte): UnicodeChar;
var
  I: Integer;
begin
  // Быстрая проверка цифр
  if Code <= $09 then
    Exit(UnicodeChar(Ord('0') + Code));

  // Явная обработка специальных символов
//  case Code of
//    $0A: Exit(' ');  // Пробел
//    $0B: Exit('!');  // Восклицательный знак
//    $0C: Exit(';');  // Точка с запятой
//    $0D: Exit(':');  // Двоеточие
//    $0E: Exit('?');  // Вопросительный знак
//    $0F: Exit('.');  // Точка
//  end;

  // Поиск в таблице
  for I := 0 to High(RusCharMap) do
    if RusCharMap[I].Code = Code then
      Exit(RusCharMap[I].Char);

  // Неизвестный код - возвращаем '?'
  Result := '?';
end;

procedure CheckCodeConflicts;
var
  I, J: Integer;
begin
  for I := 0 to High(RusCharMap) do
    for J := I+1 to High(RusCharMap) do
      if RusCharMap[I].Code = RusCharMap[J].Code then
        raise Exception.CreateFmt(
          'Конфликт кодов: %s (%x) и %s (%x)',
          [RusCharMap[I].Char, RusCharMap[I].Code,
           RusCharMap[J].Char, RusCharMap[J].Code]);
end;

initialization
  CheckCodeConflicts;
end.