unit rustringunit;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}
{$MODESWITCH ADVANCEDRECORDS}
{$MODESWITCH UNICODESTRINGS}
{$CODEPAGE UTF8}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  Classes, SysUtils, LazUTF8, LazUnicode, rustringcoreunit,Math;

const
  RUS_FILE_MAGIC = 'RUSv2';

type
  Rustring = array of Byte;
  TRusFile = array of Rustring; // Тип для файла (массив строк)

function UTF8ToRus(const S: UTF8String): Rustring;
function RusToUTF8(const R: Rustring): UTF8String;
function IsRusLetter(Code: Byte): Boolean;

// Файловые операции
function LoadRusFile(const FileName: string): TRusFile;
procedure SaveRusFile(const FileName: string; const FileData: TRusFile);
function UTF8FileToRus(const FileName: string): TRusFile;
procedure RusFileToUTF8(const FileName: string; const FileData: TRusFile);

implementation

function UTF8ToRus(const S: UTF8String): Rustring;
var
  UStr: UnicodeString;
  I: Integer;
  Code: Byte;
begin
  UStr := UTF8ToUTF16(S);
  SetLength(Result, Length(UStr));
  for I := 1 to Length(UStr) do
  begin
    Code := FindCharCode(UStr[I]);
    Result[I-1] := Code;  // Уже содержит $FF для неизвестных символов
  end;
end;

function RusToUTF8(const R: Rustring): UTF8String;
var
  UStr: UnicodeString;
  I: Integer;
begin
  SetLength(UStr, Length(R));
  for I := 0 to High(R) do
    UStr[I+1] := FindCharByCode(R[I]);
  Result := UTF16ToUTF8(UStr);
end;

function IsRusLetter(Code: Byte): Boolean;
begin
  // Русские строчные: а-я ($10-$30)
  // Русские прописные: А-Я ($31-$51)
  Result := ((Code >= $10) and (Code <= $30)) or 
            ((Code >= $31) and (Code <= $51));
end;

procedure SaveRusFile(const FileName: string; const FileData: TRusFile);
var
  F: File;
  I, J: Integer;
  Header: array[0..4] of Char;
  Count: LongInt;
  Len: LongInt;
  Checksum: LongWord;
begin
  Assign(F, FileName);
  Rewrite(F, 1);
  try
    // Заголовок файла
    Header := RUS_FILE_MAGIC;
    BlockWrite(F, Header, SizeOf(Header));
    
    // Количество строк
    Count := Length(FileData);
    BlockWrite(F, Count, SizeOf(Count));
    
    // Контрольная сумма
    Checksum := 0;
    for I := 0 to High(FileData) do
      Checksum := Checksum + Length(FileData[I]);
    BlockWrite(F, Checksum, SizeOf(Checksum));
    
    // Данные строк
    for I := 0 to High(FileData) do
    begin
      Len := Length(FileData[I]);
      BlockWrite(F, Len, SizeOf(Len));
      if Len > 0 then
        BlockWrite(F, FileData[I][0], Len);
    end;
  finally
    Close(F);
  end;
end;

function LoadRusFile(const FileName: string): TRusFile;
var
  F: File;
  Header: array[0..4] of Char;
  Count: LongInt;
  Len: LongInt;
  I: Integer;
  StoredChecksum, ActualChecksum: LongWord;
begin
  Assign(F, FileName);
  Reset(F, 1);
  try
    // Проверка заголовка
    BlockRead(F, Header, SizeOf(Header));
    if Header <> RUS_FILE_MAGIC then
      raise Exception.Create('Неверный формат файла');
    
    // Чтение количества строк
    BlockRead(F, Count, SizeOf(Count));
    SetLength(Result, Count);
    
    // Чтение контрольной суммы
    BlockRead(F, StoredChecksum, SizeOf(StoredChecksum));
    
    // Чтение данных с проверкой
    ActualChecksum := 0;
    for I := 0 to Count - 1 do
    begin
      BlockRead(F, Len, SizeOf(Len));
      SetLength(Result[I], Len);
      if Len > 0 then
        BlockRead(F, Result[I][0], Len);
      ActualChecksum := ActualChecksum + Len;
    end;
    
    // Проверка контрольной суммы
    if ActualChecksum <> StoredChecksum then
      raise Exception.Create('Ошибка целостности данных');
      
  finally
    Close(F);
  end;
end;

function UTF8FileToRus(const FileName: string): TRusFile;
var
  UTF8Lines: TStringList;
  I: Integer;
begin
  UTF8Lines := TStringList.Create;
  try
    UTF8Lines.LoadFromFile(FileName);
    SetLength(Result, UTF8Lines.Count);
    for I := 0 to UTF8Lines.Count - 1 do
      Result[I] := UTF8ToRus(UTF8Lines[I]);
  finally
    UTF8Lines.Free;
  end;
end;

procedure RusFileToUTF8(const FileName: string; const FileData: TRusFile);
var
  UTF8Lines: TStringList;
  I: Integer;
begin
  UTF8Lines := TStringList.Create;
  try
    for I := 0 to High(FileData) do
      UTF8Lines.Add(RusToUTF8(FileData[I]));
    UTF8Lines.SaveToFile(FileName);
  finally
    UTF8Lines.Free;
  end;
end;

end.