unit widgetsunit;

{
    XWidget.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}

interface

uses
  SysUtils, Classes, Unix, x, xlib, xutil;

type
  // Базовый класс для всех виджетов
  TWidget = class
  private
    FVisible: Boolean;
    FOnClick: TNotifyEvent;
  public
    constructor Create;
    procedure Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC); virtual;
    procedure HandleEvent(Event: TXEvent); virtual;
    property Visible: Boolean read FVisible write FVisible;
    property OnClick: TNotifyEvent read FOnClick write FOnClick;
  end;

  // Окно, содержащее другие виджеты
  TWindowWidget = class(TWidget)
  private
    FWidgets: TList;
  public
    constructor Create;
    destructor Destroy; override;
    procedure AddWidget(Widget: TWidget);
    procedure Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC); override;
    procedure HandleEvent(Event: TXEvent); override;
  end;

  // Кнопка
  TButton = class(TWidget)
  private
    FCaption: string;
    FX, FY, FWidth, FHeight: Integer;
  public
    constructor Create(ACaption: string; AX, AY, AWidth, AHeight: Integer);
    procedure Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC); override;
    procedure HandleEvent(Event: TXEvent); override;
  end;

  // Текстовый элемент
  TLabel = class(TWidget)
  private
    FText: string;
    FX, FY: Integer;
  public
    constructor Create(AText: string; AX, AY: Integer);
    procedure Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC); override;
  end;

  // Основной цикл обработки событий
  TEventLoop = class
  private
    FDisplay: PDisplay;
    FRootWindow: TWindowWidget;
    FGc: TGC;
    FWindow: TWindow;
    FShouldQuit: Boolean;
  public
    constructor Create(ADisplay: PDisplay; ARootWindow: TWindowWidget; AGc: TGC; AWindow: TWindow);
    procedure Run;
    property ShouldQuit: Boolean read FShouldQuit;
  end;

  // Класс для создания и управления окном
  TWindowManager = class
  private
    FDisplay: PDisplay;
    FScreen: Integer;
    FRootWindow: TWindow;
    FWindow: TWindow;
    FGc: TGC;
    FAtomDeleteWindow: TAtom;
  public
    constructor Create;
    destructor Destroy; override;
    procedure SetupWindow;
    property Display: PDisplay read FDisplay;
    property Window: TWindow read FWindow;
    property Gc: TGC read FGc;
  end;

implementation

constructor TWidget.Create;
begin
  FVisible := True;
end;

procedure TWidget.Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC);
begin
  // Базовая реализация (ничего не делает)
end;

procedure TWidget.HandleEvent(Event: TXEvent);
begin
  // Базовая обработка событий (можно переопределить в потомках)
end;

constructor TWindowWidget.Create;
begin
  inherited Create;
  FWidgets := TList.Create;
end;

destructor TWindowWidget.Destroy;
begin
  FWidgets.Free;
  inherited Destroy;
end;

procedure TWindowWidget.AddWidget(Widget: TWidget);
begin
  FWidgets.Add(Widget);
end;

procedure TWindowWidget.Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC);
var
  I: Integer;
begin
  if not Visible then Exit;
  for I := 0 to FWidgets.Count - 1 do
    TWidget(FWidgets[I]).Draw(ADisplay, AWindow, AGc);
end;

procedure TWindowWidget.HandleEvent(Event: TXEvent);
var
  I: Integer;
begin
  for I := 0 to FWidgets.Count - 1 do
    TWidget(FWidgets[I]).HandleEvent(Event);
end;

constructor TButton.Create(ACaption: string; AX, AY, AWidth, AHeight: Integer);
begin
  inherited Create;
  FCaption := ACaption;
  FX := AX;
  FY := AY;
  FWidth := AWidth;
  FHeight := AHeight;
end;

procedure TButton.Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC);
begin
  if not Visible then Exit;
  WriteLn('Drawing button at (', FX, ', ', FY, ') with size (', FWidth, ', ', FHeight, ')');
  if (FX < 0) or (FY < 0) or (FWidth <= 0) or (FHeight <= 0) then
  begin
    WriteLn('Error: Invalid button coordinates or size');
    Exit;
  end;
  XDrawRectangle(ADisplay, AWindow, AGc, FX, FY, FWidth, FHeight);
  XDrawString(ADisplay, AWindow, AGc, FX + 10, FY + 20, PChar(FCaption), Length(FCaption));
end;

procedure TButton.HandleEvent(Event: TXEvent);
begin
  if (Event._type = ButtonPress) and
     (Event.xbutton.x >= FX) and (Event.xbutton.x <= FX + FWidth) and
     (Event.xbutton.y >= FY) and (Event.xbutton.y <= FY + FHeight) then
  begin
    WriteLn('Button clicked!');
    if Assigned(FOnClick) then
      FOnClick(Self);
  end;
end;

constructor TLabel.Create(AText: string; AX, AY: Integer);
begin
  inherited Create;
  FText := AText;
  FX := AX;
  FY := AY;
end;

procedure TLabel.Draw(ADisplay: PDisplay; AWindow: TWindow; AGc: TGC);
begin
  if not Visible then Exit;
  WriteLn('Drawing label at (', FX, ', ', FY, ')');
  XDrawString(ADisplay, AWindow, AGc, FX, FY, PChar(FText), Length(FText));
end;

constructor TEventLoop.Create(ADisplay: PDisplay; ARootWindow: TWindowWidget; AGc: TGC; AWindow: TWindow);
begin
  FDisplay := ADisplay;
  FRootWindow := ARootWindow;
  FGc := AGc;
  FWindow := AWindow;
  FShouldQuit := False;
end;

procedure TEventLoop.Run;
var
  Event: TXEvent;
  AtomDeleteWindow: TAtom;
begin
  WriteLn('Event loop started');
  AtomDeleteWindow := XInternAtom(FDisplay, 'WM_DELETE_WINDOW', False);
  XSetWMProtocols(FDisplay, FWindow, @AtomDeleteWindow, 1);

  while not FShouldQuit do
  begin
    XNextEvent(FDisplay, @Event);
    WriteLn('Event received: ', Event._type);
    case Event._type of
      Expose:
        begin
          WriteLn('Expose event received');
          FRootWindow.Draw(FDisplay, FWindow, FGc);
        end;
      ClientMessage:
        begin
          if (Event.xclient.message_type = XInternAtom(FDisplay, 'WM_PROTOCOLS', False)) and
             (Event.xclient.data.l[0] = AtomDeleteWindow) then
          begin
            WriteLn('Window close event received');
            FShouldQuit := True;
          end;
        end;
      ButtonPress:
        FRootWindow.HandleEvent(Event);
    end;
    XFlush(FDisplay); // Обновляем экран
  end;
end;

constructor TWindowManager.Create;
begin
  FDisplay := XOpenDisplay(nil);
  if FDisplay = nil then
  begin
    WriteLn('Error: Unable to open display');
    Halt(1);
  end;

  FScreen := DefaultScreen(FDisplay);
  FRootWindow := RootWindow(FDisplay, FScreen);
  FWindow := 0;
  FGc := nil;
  FAtomDeleteWindow := 0;
end;

destructor TWindowManager.Destroy;
begin
  if FGc <> nil then
    XFreeGC(FDisplay, FGc);
  if FWindow <> 0 then
    XDestroyWindow(FDisplay, FWindow);
  if FDisplay <> nil then
    XCloseDisplay(FDisplay);
  inherited Destroy;
end;

procedure TWindowManager.SetupWindow;
begin
  // Создание окна
  FWindow := XCreateSimpleWindow(FDisplay, FRootWindow, 100, 100, 400, 300, 1,
                                 BlackPixel(FDisplay, FScreen), WhitePixel(FDisplay, FScreen));
  if FWindow = 0 then
  begin
    WriteLn('Error: Unable to create window');
    Halt(1);
  end;

  // Настройка окна для получения событий
  XSelectInput(FDisplay, FWindow, ExposureMask or ButtonPressMask or KeyPressMask);

  // Отображение окна
  XMapWindow(FDisplay, FWindow);

  // Создание графического контекста
  FGc := XCreateGC(FDisplay, FWindow, 0, nil);
  if FGc = nil then
  begin
    WriteLn('Error: Unable to create graphics context');
    Halt(1);
  end;

  // Настройка протокола закрытия окна
  FAtomDeleteWindow := XInternAtom(FDisplay, 'WM_DELETE_WINDOW', False);
  XSetWMProtocols(FDisplay, FWindow, @FAtomDeleteWindow, 1);
end;

end.