program XCBGui;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

uses
  xcbunit, ctypes, sysutils;

var
  c: PXCBConnection;
  screen: PXCBScreen;
  gc: TXCBGContext;
  win, back_pixmap: TXCBDrawable;

type
  TButtonCallback = procedure(b_id: cint);
  
  TButton = record
    id: TXCBDrawable;
    label_text: array[0..99] of char;
    label_orig, label_w, label_h: char;
    drag_state: cint;
    offset, origin: array[0..1] of cuint32;
    click_callback: TButtonCallback;
  end;

const
  BUTTON_HEAP_SIZE = 20;

var
  button_heap_counter: cint = 0;
  button_heap: array[0..BUTTON_HEAP_SIZE - 1] of TButton;

function AllocButton(b: TButton): PButton;
begin
  if button_heap_counter < BUTTON_HEAP_SIZE then
  begin
    button_heap[button_heap_counter] := b;
    Writeln('Registered button: ', b.id, ' to ', button_heap_counter);
    Result := @button_heap[button_heap_counter];
    Inc(button_heap_counter);
  end
  else
  begin
    Writeln('Error: can''t allocate button, out of space');
    Halt(1);
  end;
end;

procedure CreateWindow;
var
  mask: cuint32;
  values: array[0..2] of cuint32;
  fill: TXCBGContext;
begin
  back_pixmap := xcb_generate_id(c);
  xcb_create_pixmap(c, screen^.root_depth, back_pixmap, screen^.root, 800, 800);

  fill := xcb_generate_id(c);
  mask := XCB_GC_FOREGROUND;
  values[0] := screen^.white_pixel;
  xcb_create_gc(c, fill, back_pixmap, mask, @values);

  gc := xcb_generate_id(c);
  mask := XCB_GC_FOREGROUND or XCB_GC_BACKGROUND or XCB_GC_GRAPHICS_EXPOSURES;
  values[0] := screen^.black_pixel;
  values[1] := screen^.white_pixel;
  values[2] := 0;
  xcb_create_gc(c, gc, screen^.root, mask, @values);

  win := xcb_generate_id(c);
  mask := XCB_CW_BACK_PIXMAP or XCB_CW_EVENT_MASK;
  values[0] := back_pixmap;
  values[1] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_BUTTON_PRESS or 
               XCB_EVENT_MASK_STRUCTURE_NOTIFY or XCB_EVENT_MASK_BUTTON_RELEASE or 
               XCB_EVENT_MASK_KEY_PRESS or XCB_EVENT_MASK_BUTTON_MOTION;

  xcb_create_window(c, screen^.root_depth, win, screen^.root, 0, 0, 300, 300, 0,
                    XCB_WINDOW_CLASS_INPUT_OUTPUT, screen^.root_visual, mask, @values);

  xcb_map_window(c, win);
  xcb_poly_fill_rectangle(c, back_pixmap, fill, 1, @xcb_rectangle_t(XCBRectangle(x: 0, y: 0, width: 800, height: 800)));
end;

procedure EventLoop;
var
  e: PXCBGenericEvent;
begin
  while True do
  begin
    e := xcb_wait_for_event(c);
    if e = nil then
      Break;
    case e^.response_type and $7F of
      XCB_EXPOSE: Writeln('Expose Event');
      XCB_BUTTON_PRESS: Writeln('Button Press Event');
      XCB_BUTTON_RELEASE: Writeln('Button Release Event');
      XCB_KEY_PRESS: Writeln('Key Press Event');
    end;
    FreeMem(e);
  end;
end;

begin
  c := xcb_connect(nil, nil);
  screen := xcb_setup_roots_iterator(xcb_get_setup(c)).data;
  CreateWindow;
  xcb_flush(c);
  EventLoop;
  Writeln('bye!');
end.
