#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <xcb/xcb.h>

// Константы для проверки
#define XCB_CW_BACK_PIXEL 2
#define XCB_CW_EVENT_MASK 2048
#define XCB_GC_FOREGROUND 4
#define XCB_GC_BACKGROUND 8
#define XCB_GC_GRAPHICS_EXPOSURES 65536
#define XCB_EVENT_MASK_EXPOSURE 32768
#define XCB_EVENT_MASK_BUTTON_PRESS 4
#define XCB_COPY_FROM_PARENT 0
#define XCB_WINDOW_CLASS_INPUT_OUTPUT 1

// Структура для виджета
typedef struct {
    int x, y;
    int width, height;
    char *text;
    int is_rtl;
} Widget;

// Функция для отрисовки виджета
void draw_widget(xcb_connection_t *conn, xcb_window_t window, xcb_gcontext_t gc, Widget *widget) {
    if (widget == NULL) return;

    // Очистка области виджета
    xcb_rectangle_t rect = {widget->x, widget->y, widget->width, widget->height};
    uint32_t values[1];
    values[0] = 0xFFFFFF; // Белый цвет фона
    xcb_change_gc(conn, gc, XCB_GC_FOREGROUND, values);
    xcb_poly_fill_rectangle(conn, window, gc, 1, &rect);

    // Установка цвета текста (чёрный)
    values[0] = 0x000000; // Чёрный цвет текста
    xcb_change_gc(conn, gc, XCB_GC_FOREGROUND, values);

    // Отрисовка текста
    xcb_poly_text_8(conn, window, gc, widget->x + 5, widget->y + 20, strlen(widget->text), widget->text);
    xcb_flush(conn);

    printf("Widget drawn at (%d, %d) with text: %s\n", widget->x, widget->y, widget->text);
}

int main() {
    xcb_connection_t *conn;
    xcb_screen_t *screen;
    xcb_window_t window;
    xcb_gcontext_t gc;
    xcb_generic_event_t *event;
    uint32_t mask, values[2];

    // Подключение к X-серверу
    printf("Connecting to X server...\n");
    conn = xcb_connect(NULL, NULL);
    if (xcb_connection_has_error(conn)) {
        fprintf(stderr, "Error: Unable to connect to X server\n");
        return 1;
    }
    printf("Connected to X server\n");

    // Получение экрана
    printf("Getting screen...\n");
    screen = xcb_setup_roots_iterator(xcb_get_setup(conn)).data;
    if (screen == NULL) {
        fprintf(stderr, "Error: Unable to get screen\n");
        return 1;
    }
    printf("Screen obtained\n");

    // Создание окна
    printf("Creating window...\n");
    window = xcb_generate_id(conn);
    mask = XCB_CW_BACK_PIXEL | XCB_CW_EVENT_MASK;
    values[0] = screen->white_pixel; // Цвет фона (белый)
    values[1] = XCB_EVENT_MASK_EXPOSURE | XCB_EVENT_MASK_BUTTON_PRESS;
    xcb_create_window(conn, XCB_COPY_FROM_PARENT, window, screen->root,
                      100, 100, 400, 300, 1, XCB_WINDOW_CLASS_INPUT_OUTPUT,
                      screen->root_visual, mask, values);
    printf("Window created\n");

    // Отображение окна
    printf("Mapping window...\n");
    xcb_map_window(conn, window);
    xcb_flush(conn);
    printf("Window mapped\n");

    // Создание графического контекста
    printf("Creating graphics context...\n");
    gc = xcb_generate_id(conn);
    mask = XCB_GC_FOREGROUND | XCB_GC_BACKGROUND | XCB_GC_GRAPHICS_EXPOSURES;
    values[0] = screen->black_pixel; // Цвет текста (чёрный)
    values[1] = screen->white_pixel; // Цвет фона (белый)
    xcb_create_gc(conn, gc, window, mask, values);
    printf("Graphics context created\n");

    // Создание виджетов
    printf("Creating widgets...\n");
    Widget label1 = {50, 50, 200, 30, "Привет, мир!", 0};
    Widget label2 = {50, 100, 200, 30, "שלום", 1}; // Иврит (RTL)
    Widget button = {150, 150, 100, 30, "Click Me", 0};
    printf("Widgets created\n");

    // Основной цикл обработки событий
    printf("Entering event loop...\n");
    while ((event = xcb_wait_for_event(conn)) != NULL) {
        printf("Event received: type=%d\n", event->response_type & ~0x80);
        switch (event->response_type & ~0x80) {
            case XCB_EXPOSE:
                printf("Expose event received, redrawing window...\n");
                draw_widget(conn, window, gc, &label1);
                draw_widget(conn, window, gc, &label2);
                draw_widget(conn, window, gc, &button);
                xcb_flush(conn);
                break;
            case XCB_BUTTON_PRESS:
                printf("ButtonPress event received\n");
                break;
            case XCB_CLIENT_MESSAGE:
                printf("Window close request received\n");
                free(event);
                goto cleanup;
            default:
                printf("Unknown event received: type=%d, full response_type=%d\n",
                       event->response_type & ~0x80, event->response_type);
                break;
        }
        free(event);
    }

cleanup:
    // Очистка
    printf("Cleaning up...\n");
    xcb_free_gc(conn, gc);
    xcb_disconnect(conn);
    printf("Cleanup complete\n");
    return 0;
}