program icewm_session;

{
    Session.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}
uses
  SysUtils, Classes, Unix, IniFiles, Process, dbus;

type
  TSessionManager = class
  private
    FConfigFile: string;
    FWindowManager: string;
    FStartupPrograms: TStringList;
    FSessionLog: string;
    FDBusConnection: PDBusConnection;
  public
    constructor Create;
    destructor Destroy; override;
    procedure LoadConfig;
    procedure StartWindowManager;
    procedure StartAdditionalPrograms;
    procedure SaveSessionState;
    procedure RestoreSessionState;
    procedure HandleSessionEnd;
    procedure InitializeDBus;
    procedure HandleDBusMessages;
    procedure RestoreWindows;
    procedure UpdateConfig(const Key, Value: string);
  end;

constructor TSessionManager.Create;
begin
  FConfigFile := GetUserDir + '.config/icewm-session.conf';
  FWindowManager := 'icewm';
  FStartupPrograms := TStringList.Create;
  FSessionLog := GetUserDir + '.icewm/session.log';
  FDBusConnection := nil;
end;

destructor TSessionManager.Destroy;
begin
  FStartupPrograms.Free;
  if FDBusConnection <> nil then
    dbus_connection_unref(FDBusConnection);
  inherited Destroy;
end;

procedure TSessionManager.LoadConfig;
var
  Ini: TIniFile;
begin
  Ini := TIniFile.Create(FConfigFile);
  try
    FWindowManager := Ini.ReadString('Session', 'WindowManager', 'icewm');
    Ini.ReadSectionValues('Startup', FStartupPrograms);
  finally
    Ini.Free;
  end;
end;

procedure TSessionManager.StartWindowManager;
var
  Process: TProcess;
begin
  Process := TProcess.Create(nil);
  try
    Process.Executable := FWindowManager;
    Process.Options := [poWaitOnExit];
    Process.Execute;
  finally
    Process.Free;
  end;
end;

procedure TSessionManager.StartAdditionalPrograms;
var
  I: Integer;
  Process: TProcess;
begin
  for I := 0 to FStartupPrograms.Count - 1 do
  begin
    Process := TProcess.Create(nil);
    try
      Process.Executable := FStartupPrograms.ValueFromIndex[I];
      Process.Options := [poDetached];
      Process.Execute;
    finally
      Process.Free;
    end;
  end;
end;

procedure TSessionManager.SaveSessionState;
var
  LogFile: TextFile;
  Process: TProcess;
  Output: string;
begin
  AssignFile(LogFile, FSessionLog);
  Rewrite(LogFile);
  try
    // Сохраняем список открытых окон
    Process := TProcess.Create(nil);
    try
      Process.Executable := 'wmctrl';
      Process.Parameters.Add('-l');
      Process.Options := [poUsePipes, poWaitOnExit];
      Process.Execute;

      // Читаем вывод wmctrl
      SetLength(Output, 1024);
      SetLength(Output, Process.Output.Read(Output[1], Length(Output)));
      WriteLn(LogFile, Output);
    finally
      Process.Free;
    end;

    WriteLn(LogFile, 'Session state saved at ', DateTimeToStr(Now));
  finally
    CloseFile(LogFile);
  end;
end;

procedure TSessionManager.RestoreSessionState;
var
  LogFile: TextFile;
  Line: string;
begin
  if FileExists(FSessionLog) then
  begin
    AssignFile(LogFile, FSessionLog);
    Reset(LogFile);
    try
      while not EOF(LogFile) do
      begin
        ReadLn(LogFile, Line);
        WriteLn('Restoring session state: ', Line);
      end;
    finally
      CloseFile(LogFile);
    end;
  end;
end;

procedure TSessionManager.RestoreWindows;
var
  LogFile: TextFile;
  Line: string;
  Process: TProcess;
begin
  if FileExists(FSessionLog) then
  begin
    AssignFile(LogFile, FSessionLog);
    Reset(LogFile);
    try
      while not EOF(LogFile) do
      begin
        ReadLn(LogFile, Line);
        if Pos('0x', Line) = 1 then
        begin
          // Восстанавливаем окно
          Process := TProcess.Create(nil);
          try
            Process.Executable := 'wmctrl';
            Process.Parameters.Add('-ia');
            Process.Parameters.Add(Copy(Line, 1, Pos(' ', Line) - 1));
            Process.Options := [poWaitOnExit];
            Process.Execute;
          finally
            Process.Free;
          end;
        end;
      end;
    finally
      CloseFile(LogFile);
    end;
  end;
end;

procedure TSessionManager.HandleSessionEnd;
begin
  WriteLn('Ending session...');
  SaveSessionState;
  FpSystem('pkill icewmbg');
  FpSystem('pkill icewmtray');
  WriteLn('Session ended.');
end;

procedure TSessionManager.InitializeDBus;
var
  Error: DBusError;
begin
  dbus_error_init(@Error);
  FDBusConnection := dbus_bus_get(DBUS_BUS_SESSION, @Error);
  if dbus_error_is_set(@Error) <> 0 then
  begin
    WriteLn('Error: Failed to connect to D-Bus: ', Error.message);
    dbus_error_free(@Error);
    Halt(1);
  end;

  // Регистрируем интерфейс на D-Bus
  dbus_bus_request_name(FDBusConnection, 'org.icewm.SessionManager', DBUS_NAME_FLAG_REPLACE_EXISTING, @Error);
  if dbus_error_is_set(@Error) <> 0 then
  begin
    WriteLn('Error: Failed to register D-Bus name: ', Error.message);
    dbus_error_free(@Error);
    Halt(1);
  end;
end;

procedure TSessionManager.HandleDBusMessages;
var
  Message: PDBusMessage;
  Reply: PDBusMessage;
  Iter: DBusMessageIter;
  Error: DBusError;
  Key, Value: PChar;
begin
  dbus_error_init(@Error);
  Message := dbus_connection_pop_message(FDBusConnection);
  while Message <> nil do
  begin
    if Boolean(dbus_message_is_method_call(Message, 'org.icewm.SessionManager', 'Restart')) then
    begin
      // Перезапуск сессии
      HandleSessionEnd;
      StartWindowManager;
      Reply := dbus_message_new_method_return(Message);
      dbus_connection_send(FDBusConnection, Reply, nil);
      dbus_message_unref(Reply);
    end
    else if Boolean(dbus_message_is_method_call(Message, 'org.icewm.SessionManager', 'Shutdown')) then
    begin
      // Завершение сессии
      HandleSessionEnd;
      Reply := dbus_message_new_method_return(Message);
      dbus_connection_send(FDBusConnection, Reply, nil);
      dbus_message_unref(Reply);
      Halt(0); // Завершаем программу
    end
    else if Boolean(dbus_message_is_method_call(Message, 'org.icewm.SessionManager', 'SetConfig')) then
    begin
      // Изменение конфигурации
      dbus_message_iter_init(Message, @Iter);
      dbus_message_iter_get_basic(@Iter, @Key);
      dbus_message_iter_next(@Iter);
      dbus_message_iter_get_basic(@Iter, @Value);

      // Обновляем конфигурацию
      UpdateConfig(Key, Value);

      Reply := dbus_message_new_method_return(Message);
      dbus_connection_send(FDBusConnection, Reply, nil);
      dbus_message_unref(Reply);
    end;
    dbus_message_unref(Message);
    Message := dbus_connection_pop_message(FDBusConnection);
  end;
end;

procedure TSessionManager.UpdateConfig(const Key, Value: string);
var
  Ini: TIniFile;
begin
  Ini := TIniFile.Create(FConfigFile);
  try
    Ini.WriteString('Session', Key, Value);
  finally
    Ini.Free;
  end;
end;

var
  SessionManager: TSessionManager;

begin
  SessionManager := TSessionManager.Create;
  try
    // Загружаем конфигурацию
    SessionManager.LoadConfig;

    // Инициализируем D-Bus
    SessionManager.InitializeDBus;

    // Восстанавливаем состояние сессии
    SessionManager.RestoreSessionState;

    // Запускаем дополнительные программы
    SessionManager.StartAdditionalPrograms;

    // Восстанавливаем открытые окна
    SessionManager.RestoreWindows;

    // Запускаем оконный менеджер
    SessionManager.StartWindowManager;

    // Обрабатываем сообщения D-Bus
    while True do
    begin
      SessionManager.HandleDBusMessages;
      Sleep(100); // Ожидаем 100 мс перед следующей проверкой
    end;

    // Завершаем сессию
    SessionManager.HandleSessionEnd;
  finally
    SessionManager.Free;
  end;
end.