program icewmbg;

{
    BG for *wm.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}
uses
  SysUtils, Classes, Unix, BaseUnix, IniFiles, Process, dbus;

type
  TBackgroundManager = class
  private
    FBackgroundImages: TStringList;
    FCurrentImageIndex: Integer;
    FCycleInterval: Integer; // Интервал смены фона в миллисекундах
    FShuffle: Boolean;       // Перемешивать ли изображения
    FDisplayMode: string;    // Режим отображения (--bg-fill, --bg-center, и т.д.)
    FTransparencyEnabled: Boolean; // Включена ли прозрачность
    FTransparencyLevel: Integer;   // Уровень прозрачности (0-100)
    FSlideShowEnabled: Boolean;    // Включено ли слайд-шоу
    FSlideShowDelay: Integer;      // Задержка между слайдами в миллисекундах
    FMonitorCount: Integer;        // Количество мониторов
    FAnimationEnabled: Boolean;    // Включены ли анимации
    FAnimationPath: string;        // Путь к анимации
    FDBusConnection: PDBusConnection; // Соединение с D-Bus
  public
    constructor Create;
    destructor Destroy; override;
    procedure LoadConfig(const ConfigFile: string);
    procedure LoadBackgroundImages(const Path: string);
    procedure SetBackground(const ImagePath: string; MonitorIndex: Integer);
    procedure StartCycle;
    procedure ShuffleImages;
    procedure SetTransparency(Enabled: Boolean; Level: Integer);
    procedure DetectMonitors;
    procedure StartSlideShow;
    procedure StartAnimation;
    procedure SendWMCommand(const Command: string);
    procedure InitializeDBus;
    procedure HandleDBusMessages;
    procedure KillOldFehProcesses;
  end;

constructor TBackgroundManager.Create;
begin
  FBackgroundImages := TStringList.Create;
  FCurrentImageIndex := 0;
  FCycleInterval := 5000; // 5 секунд
  FShuffle := False;
  FDisplayMode := '--bg-scale'; // По умолчанию масштабирование
  FTransparencyEnabled := False;
  FTransparencyLevel := 90; // Уровень прозрачности по умолчанию
  FSlideShowEnabled := False;
  FSlideShowDelay := 5000; // Задержка между слайдами по умолчанию
  FMonitorCount := 1;      // По умолчанию один монитор
  FAnimationEnabled := False;
  FAnimationPath := '';
  FDBusConnection := nil;
end;

destructor TBackgroundManager.Destroy;
begin
  FBackgroundImages.Free;
  if FDBusConnection <> nil then
    dbus_connection_unref(FDBusConnection);
  KillOldFehProcesses;
  inherited Destroy;
end;

procedure TBackgroundManager.KillOldFehProcesses;
begin
  FpSystem('pkill feh');
end;

procedure TBackgroundManager.LoadConfig(const ConfigFile: string);
var
  Ini: TIniFile;
begin
  Ini := TIniFile.Create(ConfigFile);
  try
    FCycleInterval := Ini.ReadInteger('Settings', 'CycleInterval', 5000);
    FShuffle := Ini.ReadBool('Settings', 'Shuffle', False);
    FDisplayMode := Ini.ReadString('Settings', 'DisplayMode', '--bg-scale');
    FTransparencyEnabled := Ini.ReadBool('Settings', 'TransparencyEnabled', False);
    FTransparencyLevel := Ini.ReadInteger('Settings', 'TransparencyLevel', 90);
    FSlideShowEnabled := Ini.ReadBool('Settings', 'SlideShowEnabled', False);
    FSlideShowDelay := Ini.ReadInteger('Settings', 'SlideShowDelay', 5000);
    FAnimationEnabled := Ini.ReadBool('Settings', 'AnimationEnabled', False);
    FAnimationPath := Ini.ReadString('Settings', 'AnimationPath', '');
    LoadBackgroundImages(Ini.ReadString('Settings', 'BackgroundPath', ''));
  finally
    Ini.Free;
  end;
end;

procedure TBackgroundManager.LoadBackgroundImages(const Path: string);
var
  SearchRec: TSearchRec;
begin
  if FindFirst(Path + '/*', faAnyFile, SearchRec) = 0 then
  begin
    repeat
      if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') then
        FBackgroundImages.Add(Path + '/' + SearchRec.Name);
    until FindNext(SearchRec) <> 0;
    FindClose(SearchRec);
  end;
end;

procedure TBackgroundManager.SetBackground(const ImagePath: string; MonitorIndex: Integer);
var
  Command: string;
  fp: TextFile;
  Output: string;
begin
  // Убиваем старые процессы feh
  KillOldFehProcesses;

  // Проверяем доступ к X-серверу
  if GetEnvironmentVariable('DISPLAY') = '' then
  begin
    WriteLn('Error: DISPLAY environment variable is not set');
    Exit;
  end;

  // Формируем команду для feh
  if FMonitorCount > 1 then
    Command := 'feh ' + FDisplayMode + ' --no-xinerama --bg-center ' + ImagePath
  else
    Command := 'feh ' + FDisplayMode + ' ' + ImagePath;

  // Запускаем команду через popen
  if popen(fp, Command, 'r') = 0 then
  begin
    // Читаем вывод команды (если нужно)
    while not EOF(fp) do
    begin
      ReadLn(fp, Output);
      WriteLn(Output); // Выводим результат в консоль (для отладки)
    end;

    // Закрываем поток
    pclose(fp);
  end
  else
  begin
    WriteLn('Error: Failed to execute feh');
  end;
end;

procedure TBackgroundManager.StartCycle;
begin
  while True do
  begin
    if FBackgroundImages.Count > 0 then
    begin
      SetBackground(FBackgroundImages[FCurrentImageIndex], 0); // Устанавливаем фон для первого монитора
      FCurrentImageIndex := (FCurrentImageIndex + 1) mod FBackgroundImages.Count;
    end;
    FpSleep(FCycleInterval div 1000); // Ожидаем указанное время
  end;
end;

procedure TBackgroundManager.ShuffleImages;
var
  I, J: Integer;
  Temp: string;
begin
  Randomize;
  for I := FBackgroundImages.Count - 1 downto 1 do
  begin
    J := Random(I + 1);
    Temp := FBackgroundImages[I];
    FBackgroundImages[I] := FBackgroundImages[J];
    FBackgroundImages[J] := Temp;
  end;
end;

procedure TBackgroundManager.SetTransparency(Enabled: Boolean; Level: Integer);
var
  PID: TPID;
  Args: array of PAnsiChar;
begin
  if Enabled then
  begin
    // Запускаем picom с настройками прозрачности
    PID := FpFork;
    if PID = 0 then
    begin
      Args := ['picom', '--backend', 'glx', '--vsync', nil];
      FpExecvp('picom', PPAnsiChar(Args));
      WriteLn('Error: Failed to execute picom');
      Halt(1);
    end
    else if PID < 0 then
    begin
      WriteLn('Error: Unable to fork');
    end;
  end
  else
  begin
    // Останавливаем picom
    FpSystem('pkill picom');
  end;
end;

procedure TBackgroundManager.DetectMonitors;
var
  Output: string;
  Process: TProcess;
begin
  Process := TProcess.Create(nil);
  try
    Process.Executable := 'xrandr';
    Process.Parameters.Add('--listmonitors');
    Process.Options := [poUsePipes, poWaitOnExit];
    Process.Execute;

    // Читаем вывод xrandr
    SetLength(Output, 1024);
    SetLength(Output, Process.Output.Read(Output[1], Length(Output)));
    FMonitorCount := Length(Output.Split(' ')) - 1; // Подсчитываем количество мониторов
  finally
    Process.Free;
  end;
end;

procedure TBackgroundManager.StartSlideShow;
var
  PID: TPID;
  Args: array of PAnsiChar;
begin
  if FSlideShowEnabled then
  begin
    PID := FpFork;
    if PID = 0 then
    begin
      Args := ['feh', '--slideshow-delay', PAnsiChar(IntToStr(FSlideShowDelay div 1000)), PAnsiChar(FDisplayMode), nil];
      FpExecvp('feh', PPAnsiChar(Args));
      WriteLn('Error: Failed to execute feh for slideshow');
      Halt(1);
    end
    else if PID < 0 then
    begin
      WriteLn('Error: Unable to fork');
    end;
  end;
end;

procedure TBackgroundManager.StartAnimation;
var
  PID: TPID;
  Args: array of PAnsiChar;
begin
  if FAnimationEnabled and (FAnimationPath <> '') then
  begin
    PID := FpFork;
    if PID = 0 then
    begin
      Args := ['xwinwrap', '-ni', '-ov', '-fs', '-s', '-st', '-sp', '-b', '-nf', '--', 'mpv', '--wid=%WID', '--loop', '--no-audio', PAnsiChar(FAnimationPath), nil];
      FpExecvp('xwinwrap', PPAnsiChar(Args));
      WriteLn('Error: Failed to execute xwinwrap for animation');
      Halt(1);
    end
    else if PID < 0 then
    begin
      WriteLn('Error: Unable to fork');
    end;
  end;
end;

procedure TBackgroundManager.SendWMCommand(const Command: string);
var
  Process: TProcess;
begin
  Process := TProcess.Create(nil);
  try
    Process.Executable := 'wmctrl';
    Process.Parameters.Add(Command);
    Process.Options := [poWaitOnExit];
    Process.Execute;
  finally
    Process.Free;
  end;
end;

procedure TBackgroundManager.InitializeDBus;
var
  Error: DBusError;
begin
  dbus_error_init(@Error);
  FDBusConnection := dbus_bus_get(DBUS_BUS_SESSION, @Error);
  if dbus_error_is_set(@Error) <> 0 then
  begin
    WriteLn('Error: Failed to connect to D-Bus: ', Error.message);
    dbus_error_free(@Error);
    Halt(1);
  end;

  // Регистрируем интерфейс на D-Bus
  dbus_bus_request_name(FDBusConnection, 'org.icewm.BackgroundManager', DBUS_NAME_FLAG_REPLACE_EXISTING, @Error);
  if dbus_error_is_set(@Error) <> 0 then
  begin
    WriteLn('Error: Failed to register D-Bus name: ', Error.message);
    dbus_error_free(@Error);
    Halt(1);
  end;
end;

procedure TBackgroundManager.HandleDBusMessages;
var
  Message: PDBusMessage;
  Reply: PDBusMessage;
  Iter: DBusMessageIter;
  Error: DBusError;
  Command: PChar;
begin
  dbus_error_init(@Error);
  Message := dbus_connection_pop_message(FDBusConnection);
  while Message <> nil do
  begin
    if Boolean(dbus_message_is_method_call(Message, 'org.icewm.BackgroundManager', 'SetBackground')) then
    begin
      dbus_message_iter_init(Message, @Iter);
      dbus_message_iter_get_basic(@Iter, @Command);
      SetBackground(Command, 0); // Устанавливаем фон для первого монитора
      Reply := dbus_message_new_method_return(Message);
      dbus_connection_send(FDBusConnection, Reply, nil);
      dbus_message_unref(Reply);
    end;
    dbus_message_unref(Message);
    Message := dbus_connection_pop_message(FDBusConnection);
  end;
end;

var
  BackgroundManager: TBackgroundManager;

procedure HandleSignal(Sig: cint); cdecl;
begin
  case Sig of
    SIGTERM, SIGINT:
      begin
        WriteLn('Exiting...');
        Halt(0);
      end;
    SIGUSR1:
      begin
        WriteLn('Shuffling images...');
        BackgroundManager.ShuffleImages;
      end;
  end;
end;

procedure Daemonize;
var
  PID: TPID;
begin
  // Шаг 1: Создаём новый процесс
  PID := FpFork;
  if PID < 0 then
  begin
    WriteLn('Error: Unable to fork');
    Halt(1);
  end;
  if PID > 0 then
    Halt(0); // Завершаем родительский процесс

  // Шаг 2: Создаём новый сеанс
  if FpSetsid < 0 then
  begin
    WriteLn('Error: Unable to create new session');
    Halt(1);
  end;

  // Шаг 3: Закрываем стандартные файловые дескрипторы
  FpClose(StdInputHandle);
  FpClose(StdOutputHandle);
  FpClose(StdErrorHandle);

  // Шаг 4: Перенаправляем вывод в /dev/null
  AssignFile(Output, '/dev/null');
  Rewrite(Output);
  AssignFile(ErrOutput, '/dev/null');
  Rewrite(ErrOutput);
end;

begin
  // Демонизируем программу
  Daemonize;

  BackgroundManager := TBackgroundManager.Create;
  try
    // Загружаем конфигурацию
    BackgroundManager.LoadConfig(GetUserDir + '.config/icewmbg.conf');

    // Определяем количество мониторов
    BackgroundManager.DetectMonitors;

    // Включаем прозрачность, если она настроена
    if BackgroundManager.FTransparencyEnabled then
      BackgroundManager.SetTransparency(True, BackgroundManager.FTransparencyLevel);

    // Перемешиваем изображения, если нужно
    if BackgroundManager.FShuffle then
      BackgroundManager.ShuffleImages;

    // Запускаем слайд-шоу, если оно включено
    if BackgroundManager.FSlideShowEnabled then
      BackgroundManager.StartSlideShow
    else
      // Начинаем цикл смены фона
      BackgroundManager.StartCycle;

    // Запускаем анимацию, если она включена
    if BackgroundManager.FAnimationEnabled then
      BackgroundManager.StartAnimation;

    // Инициализируем D-Bus
    BackgroundManager.InitializeDBus;

    // Устанавливаем обработчик сигналов
    FpSignal(SIGTERM, @HandleSignal);
    FpSignal(SIGINT, @HandleSignal);
    FpSignal(SIGUSR1, @HandleSignal);

    // Основной цикл программы
    while True do
    begin
      // Обрабатываем сообщения D-Bus
      BackgroundManager.HandleDBusMessages;

      // Ожидаем сигналов
      FpPause;
    end;
  finally
    BackgroundManager.Free;
  end;
end.