unit string4unit;

{$MODE OBJFPC}{$H+}

interface

uses
  SysUtils, LazUTF8;

type
  string4 = class
  private
    FData: array of UInt32; // UTF-32/UCS-4
    function GetLength: Integer;
    function GetChar(Index: Integer): UInt32;
    procedure SetChar(Index: Integer; Value: UInt32);
  public
    // Конструктор и деструктор
    constructor Create;
    destructor Destroy; override;

    // Преобразование из/в UTF-8
    procedure FromUTF8(const S: string);
    function ToUTF8: string;

    // Операции с символами
    property Length: Integer read GetLength;
    property Chars[Index: Integer]: UInt32 read GetChar write SetChar; default;

    // Конкатенация
    function Concat(const S: string4): string4;

    // Поддержка двунаправленного текста
    function IsRTL: Boolean; // Проверка, содержит ли строка RTL символы
    procedure Reverse; // Реверс строки (полезно для RTL текста)

    // Оптимизация
    procedure Clear; // Очистка строки
    procedure Reserve(Capacity: Integer); // Резервирование памяти
  end;

implementation

constructor string4.Create;
begin
  inherited Create;
  SetLength(FData, 0); // Инициализация массива
end;

destructor string4.Destroy;
begin
  SetLength(FData, 0); // Освобождение памяти
  inherited Destroy;
end;

procedure string4.FromUTF8(const S: string);
var
  UTF8Ptr: PChar;
  CharLen: Integer;
  CodePoint: UInt32;
begin
  SetLength(FData, 0); // Очистка массива
  UTF8Ptr := PChar(S);
  while UTF8Ptr^ <> #0 do
  begin
    CodePoint := UTF8CodepointToUnicode(UTF8Ptr, CharLen);
    if CodePoint = 0 then Break;
    SetLength(FData, system.Length(FData) + 1);
    FData[High(FData)] := CodePoint;
    Inc(UTF8Ptr, CharLen);
  end;
end;

function string4.ToUTF8: string;
var
  I: Integer;
  UTF8Char: string;
begin
  Result := '';
  for I := 0 to High(FData) do
  begin
    UTF8Char := UnicodeToUTF8(FData[I]);
    Result := Result + UTF8Char;
  end;
end;

function string4.GetLength: Integer;
begin
  Result := System.Length(FData);
end;

function string4.GetChar(Index: Integer): UInt32;
begin
  if (Index < 0) or (Index >= system.Length(FData)) then
    raise Exception.Create('Index out of bounds');
  Result := FData[Index];
end;

procedure string4.SetChar(Index: Integer; Value: UInt32);
begin
  if (Index < 0) or (Index >= system.Length(FData)) then
    raise Exception.Create('Index out of bounds');
  FData[Index] := Value;
end;

function string4.Concat(const S: string4): string4;
var
  I: Integer;
begin
  Result := string4.Create;
  SetLength(Result.FData, Self.Length + S.Length);
  for I := 0 to Self.Length - 1 do
    Result.FData[I] := Self.FData[I];
  for I := 0 to S.Length - 1 do
    Result.FData[Self.Length + I] := S.FData[I];
end;

function string4.IsRTL: Boolean;
var
  I: Integer;
begin
  Result := False;
  for I := 0 to High(FData) do
  begin
    // Проверка на RTL символы (например, арабские, иврит и т.д.)
    if (FData[I] >= $0590) and (FData[I] <= $08FF) then
    begin
      Result := True;
      Exit;
    end;
  end;
end;

procedure string4.Reverse;
var
  I, J: Integer;
  Temp: UInt32;
begin
  I := 0;
  J := High(FData);
  while I < J do
  begin
    Temp := FData[I];
    FData[I] := FData[J];
    FData[J] := Temp;
    Inc(I);
    Dec(J);
  end;
end;

procedure string4.Clear;
begin
  SetLength(FData, 0);
end;

procedure string4.Reserve(Capacity: Integer);
begin
  if Capacity > system.Length(FData) then
    SetLength(FData, Capacity);
end;

end.