program testeventloop;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}
uses
  SysUtils, xcbunit, ctypes, Unix;

const
  LC_ALL = 6;

function SetLocale(Context: Integer; Locale: PChar): PChar; cdecl; external name 'setlocale';

var
  Connection: Pxcb_connection_t;
  Screen: Pxcb_screen_t;
  Win: xcb_window_t;
  Gc: xcb_gcontext_t; // Графический контекст
  EventMask: array[0..0] of cuint32;
  Event: Pxcb_generic_event_t;
  ExposeEvent: xcb_expose_event_t;
  WM_PROTOCOLS, WM_DELETE_WINDOW: xcb_atom_t;
  Cookie: xcb_intern_atom_cookie_t;
  Reply: Pxcb_intern_atom_reply_t;

procedure HandleEvent(Event: Pxcb_generic_event_t);
var
  ClientMessageEvent: Pxcb_client_message_event_t;
  ExposeEvent: Pxcb_expose_event_t;
  Rect: xcb_rectangle_t;
begin
  case Event^.response_type and $7F of
    XCB_EXPOSE:
      begin
        ExposeEvent := Pxcb_expose_event_t(Event);
        WriteLn('Expose event received: window=', ExposeEvent^.window,
                ', x=', ExposeEvent^.x, ', y=', ExposeEvent^.y,
                ', width=', ExposeEvent^.width, ', height=', ExposeEvent^.height);

        // Очистка окна (заливка белым цветом)
        Rect.x := ExposeEvent^.x;
        Rect.y := ExposeEvent^.y;
        Rect.width := ExposeEvent^.width;
        Rect.height := ExposeEvent^.height;
        xcb_poly_fill_rectangle(Connection, Win, Gc, 1, @Rect);

        xcb_flush(Connection);
      end;
    XCB_CLIENT_MESSAGE:
      begin
        ClientMessageEvent := Pxcb_client_message_event_t(Event);
        if ClientMessageEvent^.data.data32[0] = WM_DELETE_WINDOW then
        begin
          WriteLn('Window close request received');
          Halt(0); // Корректное завершение программы
        end;
      end;
    else
      WriteLn('Unknown event type: ', Event^.response_type and $7F);
  end;
end;

begin
  // Установка локали
  WriteLn('Setting locale...');
  setlocale(LC_ALL, '');

  // Подключение к X-серверу
  WriteLn('Connecting to X server...');
  Connection := xcb_connect(nil, nil);
  if xcb_connection_has_error(Connection) <> 0 then
  begin
    WriteLn('Error: Unable to connect to X server');
    Halt(1);
  end;
  WriteLn('Connected to X server');

  // Получение экрана
  WriteLn('Getting screen...');
  Screen := xcb_setup_roots_iterator(xcb_get_setup(Connection)).data;
  if Screen = nil then
  begin
    WriteLn('Error: Unable to get screen');
    Halt(1);
  end;
  WriteLn('Screen obtained');

  // Вывод информации о экране
  WriteLn('Screen information:');
  WriteLn('  root: ', Screen^.root);
  WriteLn('  white_pixel: ', Screen^.white_pixel);
  WriteLn('  black_pixel: ', Screen^.black_pixel);
  WriteLn('  root_visual: ', Screen^.root_visual);
  WriteLn('  width_in_pixels: ', Screen^.width_in_pixels);
  WriteLn('  height_in_pixels: ', Screen^.height_in_pixels);

  // Создание окна
  WriteLn('Creating window...');
  Win := xcb_generate_id(Connection);

  // Установка маски событий
  EventMask[0] := XCB_EVENT_MASK_EXPOSURE; // Упрощённая маска событий

  xcb_create_window(
    Connection,
    XCB_COPY_FROM_PARENT,
    Win,
    Screen^.root,
    100, 100,
    400, 300,
    1,
    XCB_WINDOW_CLASS_INPUT_OUTPUT,
    Screen^.root_visual,
    XCB_CW_EVENT_MASK,
    @EventMask
  );
  WriteLn('Window created');

  // Создание графического контекста
  WriteLn('Creating graphics context...');
  Gc := xcb_generate_id(Connection);
  xcb_create_gc(
    Connection,
    Gc,
    Win,
    0, // value_mask (0 означает, что используются значения по умолчанию)
    nil // value_list (nil, так как value_mask = 0)
  );
  WriteLn('Graphics context created');

  // Получение атомов
  WriteLn('Getting atoms...');
  Cookie := xcb_intern_atom(Connection, 0, Length('WM_PROTOCOLS'), 'WM_PROTOCOLS');
  Reply := xcb_intern_atom_reply(Connection, Cookie, nil);
  if Reply <> nil then
  begin
    WM_PROTOCOLS := Reply^.atom;
    // Не освобождаем память, так как XCB управляет ею самостоятельно
  end
  else
  begin
    WriteLn('Error: Unable to get WM_PROTOCOLS atom');
    Halt(1);
  end;

  Cookie := xcb_intern_atom(Connection, 0, Length('WM_DELETE_WINDOW'), 'WM_DELETE_WINDOW');
  Reply := xcb_intern_atom_reply(Connection, Cookie, nil);
  if Reply <> nil then
  begin
    WM_DELETE_WINDOW := Reply^.atom;
    // Не освобождаем память, так как XCB управляет ею самостоятельно
  end
  else
  begin
    WriteLn('Error: Unable to get WM_DELETE_WINDOW atom');
    Halt(1);
  end;
  WriteLn('Atoms obtained');

  // Установка протокола закрытия окна
  WriteLn('Setting WM_DELETE_WINDOW protocol...');
  xcb_change_property(
    Connection,
    XCB_PROP_MODE_REPLACE,
    Win,
    WM_PROTOCOLS,
    XCB_ATOM_ATOM,
    32,
    1,
    @WM_DELETE_WINDOW
  );
  WriteLn('Protocol set');

  // Отображение окна
  WriteLn('Mapping window...');
  xcb_map_window(Connection, Win);
  xcb_flush(Connection);
  WriteLn('Window mapped');

  // Добавляем задержку перед входом в цикл событий
  WriteLn('Waiting for window to appear...');
  Sleep(1000); // Задержка 1 секунда

  // Основной цикл обработки событий
  WriteLn('Entering event loop');
  while True do
  begin
    Event := xcb_poll_for_event(Connection); // Используем xcb_poll_for_event вместо xcb_wait_for_event
    if Event <> nil then
    begin
      WriteLn('Event received: type=', Event^.response_type and $7F);
      HandleEvent(Event);
      // Не освобождаем память для события, так как XCB управляет ею самостоятельно
    end
    else
    begin
      Sleep(10); // Небольшая задержка, чтобы уменьшить нагрузку на процессор
    end;
  end;

  // Очистка
  WriteLn('Cleaning up...');
  xcb_free_gc(Connection, Gc); // Освобождаем графический контекст
  xcb_disconnect(Connection);
  WriteLn('Cleanup complete');
end.