program widgetstest;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

uses
  SysUtils, xcbunit, widgetsunit, string4unit, strutils, ctypes;

const
  LC_ALL = 6;

function SetLocale(Context: Integer; Locale: PChar): PChar; cdecl; external name 'setlocale';

type
  TEventHandler = class
  public
    procedure ButtonClick(Sender: TObject);
  end;

procedure TEventHandler.ButtonClick(Sender: TObject);
begin
  WriteLn('Button clicked!');
end;

var
  Connection: Pxcb_connection_t;
  Screen: Pxcb_screen_t;
  Win: xcb_window_t;
  Gc: xcb_gcontext_t;
  EventLoop: TEventLoop;
  MainWindow: TWindowWidget;
  Label1, Label2: TLabel;
  MyText1, MyText2: string4;
  Button: TButton; // Кнопка
  EventHandler: TEventHandler; // Обработчик событий
  EventMask: array[0..1] of cuint32; // Массив для передачи event_mask
  Values: array[0..1] of cuint32; // Массив для передачи значений в графический контекст
  WM_PROTOCOLS, WM_DELETE_WINDOW: xcb_atom_t; // Атомы для протоколов
  Cookie: xcb_intern_atom_cookie_t;
  Reply: Pxcb_intern_atom_reply_t;
  ev: xcb_expose_event_t;
  Mask: cuint32; // Переменная для маски графического контекста
  WhiteColor, BlackColor: cuint32;
  Rect: xcb_rectangle_t;

begin
  // Установка локали
  WriteLn('Setting locale...');
  setlocale(LC_ALL, '');

  // Подключение к X-серверу
  WriteLn('Connecting to X server...');
  Connection := xcb_connect(nil, nil);
  if xcb_connection_has_error(Connection) <> 0 then
  begin
    WriteLn('Error: Unable to connect to X server');
    Halt(1);
  end;
  WriteLn('Connected to X server');

  // Получение экрана
  WriteLn('Getting screen...');
  Screen := xcb_setup_roots_iterator(xcb_get_setup(Connection)).data;
  if Screen = nil then
  begin
    WriteLn('Error: Unable to get screen');
    Halt(1);
  end;
  WriteLn('Screen obtained');

  // Создание окна
  WriteLn('Creating window...');
  Win := xcb_generate_id(Connection);

  // Установка event_mask и цвета фона
  EventMask[0] := Screen^.white_pixel; // Цвет фона (белый)
  EventMask[1] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_BUTTON_PRESS or XCB_EVENT_MASK_KEY_PRESS;

  xcb_create_window(
    Connection,
    XCB_COPY_FROM_PARENT,
    Win,
    Screen^.root,
    100, 100,
    400, 300,
    1,
    XCB_WINDOW_CLASS_INPUT_OUTPUT,
    Screen^.root_visual,
    XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK,
    @EventMask[0]
  );
  WriteLn('Window created');

  // Получение атомов
  WriteLn('Getting atoms...');
  Cookie := xcb_intern_atom(Connection, 0, Length('WM_PROTOCOLS'), 'WM_PROTOCOLS');
  Reply := xcb_intern_atom_reply(Connection, Cookie, nil);
  if Reply <> nil then
  begin
    WM_PROTOCOLS := Reply^.atom;
  end
  else
  begin
    WriteLn('Error: Unable to get WM_PROTOCOLS atom');
    Halt(1);
  end;

  Cookie := xcb_intern_atom(Connection, 0, Length('WM_DELETE_WINDOW'), 'WM_DELETE_WINDOW');
  Reply := xcb_intern_atom_reply(Connection, Cookie, nil);
  if Reply <> nil then
  begin
    WM_DELETE_WINDOW := Reply^.atom;
    FreeMem(Reply);
  end
  else
  begin
    WriteLn('Error: Unable to get WM_DELETE_WINDOW atom');
    Halt(1);
  end;
  WriteLn('Atoms obtained');

  // Установка протокола закрытия окна
  WriteLn('Setting WM_DELETE_WINDOW protocol...');
  xcb_change_property(
    Connection,
    XCB_PROP_MODE_REPLACE,
    Win,
    WM_PROTOCOLS,
    XCB_ATOM_ATOM,
    32,
    1,
    @WM_DELETE_WINDOW
  );
  WriteLn('Protocol set');

  // Отображение окна
  WriteLn('Mapping window...');
  xcb_map_window(Connection, Win);

  // Принудительная отправка события Expose
  ev.response_type := XCB_EXPOSE;
  ev.window := Win;
  ev.x := 0;
  ev.y := 0;
  ev.width := 400;
  ev.height := 300;
  ev.count := 0;
  xcb_send_event(Connection, 0, Win, XCB_EVENT_MASK_EXPOSURE, PChar(@ev));
  xcb_flush(Connection);

  WriteLn('Window mapped');

  // Создание графического контекста
  WriteLn('Creating graphics context...');
  Gc := xcb_generate_id(Connection);

  // Установка цветов
  WhiteColor := Screen^.white_pixel; // Цвет фона (белый)
  BlackColor := Screen^.black_pixel; // Цвет текста (чёрный)

  // Установка цвета фона и текста
  Values[0] := BlackColor; // Цвет переднего плана (чёрный)
  Values[1] := WhiteColor; // Цвет фона (белый)

  // Создание графического контекста
  Mask := XCB_GC_FOREGROUND or XCB_GC_BACKGROUND or XCB_GC_GRAPHICS_EXPOSURES;
  xcb_create_gc(Connection, Gc, Win, Mask, @Values[0]);
  WriteLn('Graphics context created');

  // Создание главного окна
  WriteLn('Creating main window...');
  MainWindow := TWindowWidget.Create(Screen);
  if MainWindow = nil then
  begin
    WriteLn('Error: Unable to create main window');
    Halt(1);
  end;
  WriteLn('Main window created');

  // Создание текстовых элементов
  WriteLn('Creating labels...');
  MyText1 := string4.Create;
  MyText1.FromUTF8('Привет, мир!');
  Label1 := TLabel.Create(MyText1, 50, 50, Screen); // Лейбл 1
  if Label1 = nil then
  begin
    WriteLn('Error: Unable to create Label1');
    Halt(1);
  end;

  MyText2 := string4.Create;
  MyText2.FromUTF8('שלום'); // Иврит (RTL)
  Label2 := TLabel.Create(MyText2, 50, 100, Screen, True); // Лейбл 2 (RTL)
  if Label2 = nil then
  begin
    WriteLn('Error: Unable to create Label2');
    Halt(1);
  end;
  WriteLn('Labels created');

  // Создание кнопки
  WriteLn('Creating button...');
  Button := TButton.Create('Click Me', 150, 150, 100, 30, Screen); // Кнопка
  if Button = nil then
  begin
    WriteLn('Error: Unable to create Button');
    Halt(1);
  end;
  WriteLn('Button created');

  // Создание обработчика событий
  EventHandler := TEventHandler.Create;
  Button.OnClick := @EventHandler.ButtonClick; // Привязываем обработчик события

  // Добавление виджетов в главное окно
  WriteLn('Adding widgets to main window...');
  MainWindow.AddWidget(Label1);
  MainWindow.AddWidget(Label2);
  MainWindow.AddWidget(Button); // Добавляем кнопку
  WriteLn('Widgets added');

  // Основной цикл обработки событий
  WriteLn('Creating event loop...');
  EventLoop := TEventLoop.Create(Connection, MainWindow, Gc, Win);
  if EventLoop = nil then
  begin
    WriteLn('Error: Unable to create event loop');
    Halt(1);
  end;
  WriteLn('Event loop created');

  // Проверка состояния соединения перед запуском цикла
  if xcb_connection_has_error(Connection) <> 0 then
  begin
    WriteLn('Error: Connection to X server lost before event loop');
    Halt(1);
  end;

  try
    WriteLn('Running event loop...');
    EventLoop.Run;
  except
    on E: Exception do
    begin
      WriteLn('An error occurred: ', E.Message);
      Halt(1);
    end;
  end;

  // Очистка
  WriteLn('Cleaning up...');
  MyText1.Free;
  MyText2.Free;
  MainWindow.Free;
  Label1.Free;
  Label2.Free;
  Button.Free; // Освобождаем кнопку
  EventHandler.Free; // Освобождаем обработчик событий
  EventLoop.Free;
  xcb_free_gc(Connection, Gc);
  xcb_disconnect(Connection);
  WriteLn('Cleanup complete');
end.