unit widgetsunit;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

interface

uses
  SysUtils, Classes, Unix, xcbunit, string4unit, strutils;

type
  // Базовый класс для всех виджетов
  TWidget = class
  private
    FVisible: Boolean;
    FOnClick: TNotifyEvent;
  public
    constructor Create;
    procedure Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t); virtual;
    procedure HandleEvent(Event: Pxcb_generic_event_t); virtual;
    property Visible: Boolean read FVisible write FVisible;
    property OnClick: TNotifyEvent read FOnClick write FOnClick;
  end;

  // Окно, содержащее другие виджеты
  TWindowWidget = class(TWidget)
  private
    FWidgets: TList;
    FScreen: Pxcb_screen_t; // Указатель на экран
  public
    constructor Create(AScreen: Pxcb_screen_t);
    destructor Destroy; override;
    procedure AddWidget(Widget: TWidget);
    procedure Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t); override;
    procedure HandleEvent(Event: Pxcb_generic_event_t); override;
  end;

  // Текстовый элемент
  TLabel = class(TWidget)
  private
    FText: string4;
    FX, FY: Integer;
    FIsRTL: Boolean;
    FScreen: Pxcb_screen_t; // Указатель на экран
  public
    constructor Create(const AText: string4; AX, AY: Integer; AScreen: Pxcb_screen_t; AIsRTL: Boolean = False);
    procedure Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t); override;
    property Text: string4 read FText write FText;
  end;

  // Кнопка
  TButton = class(TWidget)
  private
    FText: string;
    FX, FY: Integer;
    FWidth, FHeight: Integer;
    FScreen: Pxcb_screen_t; // Указатель на экран
  public
    constructor Create(const AText: string; AX, AY, AWidth, AHeight: Integer; AScreen: Pxcb_screen_t);
    procedure Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t); override;
    procedure HandleEvent(Event: Pxcb_generic_event_t); override;
  end;

  // Основной цикл обработки событий
  TEventLoop = class
  private
    FConnection: Pxcb_connection_t;
    FRootWindow: TWindowWidget;
    FGc: xcb_gcontext_t;
    FWindow: xcb_window_t;
  public
    constructor Create(AConnection: Pxcb_connection_t; ARootWindow: TWindowWidget; AGc: xcb_gcontext_t; AWindow: xcb_window_t);
    procedure Run;
  end;

implementation

constructor TWidget.Create;
begin
  FVisible := True;
end;

procedure TWidget.Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t);
begin
  // Базовая реализация (ничего не делает)
end;

procedure TWidget.HandleEvent(Event: Pxcb_generic_event_t);
begin
  // Базовая обработка событий (можно переопределить в потомках)
end;

constructor TWindowWidget.Create(AScreen: Pxcb_screen_t);
begin
  inherited Create;
  FWidgets := TList.Create;
  FScreen := AScreen; // Сохраняем указатель на экран
end;

destructor TWindowWidget.Destroy;
begin
  FWidgets.Free;
  inherited Destroy;
end;

procedure TWindowWidget.AddWidget(Widget: TWidget);
begin
  if Widget <> nil then
    FWidgets.Add(Widget);
end;

procedure TWindowWidget.Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t);
var
  I: Integer;
  Widget: TWidget;
  Rect: xcb_rectangle_t;
  Values: array[0..1] of cuint32;
  Mask: cuint32;
begin
  if not Visible then Exit;

  // Установка цвета фона окна (белый)
  Values[0] := FScreen^.white_pixel;
  Mask := XCB_GC_FOREGROUND;
  xcb_change_gc(AConnection, AGc, Mask, @Values[0]);

  // Очистка экрана (заливка белым цветом)
  Rect.x := 0;
  Rect.y := 0;
  Rect.width := 400; // Ширина окна
  Rect.height := 300; // Высота окна
  xcb_poly_fill_rectangle(AConnection, AWindow, AGc, 1, @Rect);

  // Отрисовка всех виджетов
  for I := 0 to FWidgets.Count - 1 do
  begin
    Widget := TWidget(FWidgets[I]);
    if Widget <> nil then
    begin
      WriteLn('Drawing widget ', I);
      Widget.Draw(AConnection, AWindow, AGc);
    end
    else
      WriteLn('Error: Widget is nil at index ', I);
  end;

  // Принудительное обновление экрана
  xcb_flush(AConnection);
end;

procedure TWindowWidget.HandleEvent(Event: Pxcb_generic_event_t);
var
  I: Integer;
begin
  for I := 0 to FWidgets.Count - 1 do
    TWidget(FWidgets[I]).HandleEvent(Event);
end;

constructor TLabel.Create(const AText: string4; AX, AY: Integer; AScreen: Pxcb_screen_t; AIsRTL: Boolean);
begin
  inherited Create;
  FText := AText;
  FX := AX;
  FY := AY;
  FIsRTL := AIsRTL;
  FScreen := AScreen; // Сохраняем указатель на экран
end;

procedure TLabel.Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t);
var
  UTF8Text: string;
  Rect: xcb_rectangle_t;
  Values: array[0..1] of cuint32;
  Mask: cuint32;
begin
  if not Visible then Exit;

  // Очистка области перед отрисовкой
  Rect.x := FX;
  Rect.y := FY;
  Rect.width := 200; // Ширина области
  Rect.height := 30; // Высота области
  Values[0] := FScreen^.white_pixel; // Цвет фона (белый)
  Mask := XCB_GC_FOREGROUND;
  xcb_change_gc(AConnection, AGc, Mask, @Values[0]);
  xcb_poly_fill_rectangle(AConnection, AWindow, AGc, 1, @Rect);

  // Установка цвета текста (чёрный)
  Values[0] := FScreen^.black_pixel;
  Mask := XCB_GC_FOREGROUND;
  xcb_change_gc(AConnection, AGc, Mask, @Values[0]);

  UTF8Text := FText.ToUTF8;
  if FIsRTL then
    UTF8Text := ReverseString(UTF8Text); // Реверс текста для RTL

  // Вывод текста в окно
  xcb_poly_text_8(AConnection, AWindow, AGc, FX + 5, FY + 20, Length(UTF8Text), PChar(UTF8Text));
  xcb_flush(AConnection);
end;

constructor TButton.Create(const AText: string; AX, AY, AWidth, AHeight: Integer; AScreen: Pxcb_screen_t);
begin
  inherited Create;
  FText := AText;
  FX := AX;
  FY := AY;
  FWidth := AWidth;
  FHeight := AHeight;
  FScreen := AScreen; // Сохраняем указатель на экран
end;

procedure TButton.Draw(AConnection: Pxcb_connection_t; AWindow: xcb_window_t; AGc: xcb_gcontext_t);
var
  Rect: xcb_rectangle_t;
  Values: array[0..1] of cuint32;
  Mask: cuint32;
begin
  if not Visible then Exit;

  // Установка цвета фона кнопки (белый)
  Values[0] := FScreen^.white_pixel;
  Mask := XCB_GC_FOREGROUND;
  xcb_change_gc(AConnection, AGc, Mask, @Values[0]);

  // Отрисовка фона кнопки
  Rect.x := FX;
  Rect.y := FY;
  Rect.width := FWidth;
  Rect.height := FHeight;
  xcb_poly_fill_rectangle(AConnection, AWindow, AGc, 1, @Rect);

  // Установка цвета текста (чёрный)
  Values[0] := FScreen^.black_pixel;
  Mask := XCB_GC_FOREGROUND;
  xcb_change_gc(AConnection, AGc, Mask, @Values[0]);

  // Отрисовка текста на кнопке
  xcb_poly_text_8(AConnection, AWindow, AGc, FX + 10, FY + 20, Length(FText), PChar(FText));
end;

procedure TButton.HandleEvent(Event: Pxcb_generic_event_t);
var
  ButtonPress: Pxcb_button_press_event_t;
begin
  if Event^.response_type = XCB_BUTTON_PRESS then
  begin
    ButtonPress := Pxcb_button_press_event_t(Event);
    if (ButtonPress^.event_x >= FX) and (ButtonPress^.event_x <= FX + FWidth) and
       (ButtonPress^.event_y >= FY) and (ButtonPress^.event_y <= FY + FHeight) then
    begin
      if Assigned(OnClick) then
        OnClick(Self);
    end;
  end;
end;

constructor TEventLoop.Create(AConnection: Pxcb_connection_t; ARootWindow: TWindowWidget; AGc: xcb_gcontext_t; AWindow: xcb_window_t);
begin
  FConnection := AConnection;
  FRootWindow := ARootWindow;
  FGc := AGc;
  FWindow := AWindow;
end;

procedure TEventLoop.Run;
var
  Event: Pxcb_generic_event_t;
  ClientMessage: Pxcb_client_message_event_t;
  WM_PROTOCOLS, WM_DELETE_WINDOW: xcb_atom_t;
  Cookie: xcb_intern_atom_cookie_t;
  Reply: Pxcb_intern_atom_reply_t;
begin
  // Получение атома WM_PROTOCOLS
  Cookie := xcb_intern_atom(FConnection, 0, Length('WM_PROTOCOLS'), 'WM_PROTOCOLS');
  Reply := xcb_intern_atom_reply(FConnection, Cookie, nil);
  if Reply = nil then
  begin
    WriteLn('Error: Unable to get WM_PROTOCOLS atom');
    Halt(1);
  end;
  WM_PROTOCOLS := Reply^.atom;

  // Получение атома WM_DELETE_WINDOW
  Cookie := xcb_intern_atom(FConnection, 0, Length('WM_DELETE_WINDOW'), 'WM_DELETE_WINDOW');
  Reply := xcb_intern_atom_reply(FConnection, Cookie, nil);
  if Reply = nil then
  begin
    WriteLn('Error: Unable to get WM_DELETE_WINDOW atom');
    Halt(1);
  end;
  WM_DELETE_WINDOW := Reply^.atom;

  while True do
  begin
    // Проверка состояния соединения
    if xcb_connection_has_error(FConnection) <> 0 then
    begin
      WriteLn('Error: Connection to X server lost');
      Break;
    end;

    Event := xcb_wait_for_event(FConnection);
    if Event = nil then
    begin
      WriteLn('Error: Unable to get event');
      Break;
    end;

    case Event^.response_type and not $80 of
      XCB_EXPOSE:
        begin
          WriteLn('Expose event received, redrawing window...');
          FRootWindow.Draw(FConnection, FWindow, FGc);
          xcb_flush(FConnection);
        end;
      XCB_CONFIGURE_NOTIFY:
        begin
          WriteLn('ConfigureNotify event received, redrawing window...');
          FRootWindow.Draw(FConnection, FWindow, FGc);
          xcb_flush(FConnection);
        end;
      XCB_BUTTON_PRESS:
        begin
          WriteLn('ButtonPress event received');
          FRootWindow.HandleEvent(Event);
        end;
      XCB_CLIENT_MESSAGE:
        begin
          ClientMessage := Pxcb_client_message_event_t(Event);
          if (ClientMessage^.data.data32[0] = WM_DELETE_WINDOW) then
          begin
            WriteLn('Window close request received');
            Break;
          end;
        end;
      XCB_DESTROY_NOTIFY:
        begin
          WriteLn('Window closed');
          Break;
        end;
    end;

//    FreeMem(Event);
  end;
  WriteLn('Exiting event loop');
end;

end.