#include <xcb/xcb.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

int main() {
    // Подключаемся к X серверу
    xcb_connection_t *connection = xcb_connect(NULL, NULL);
    if (xcb_connection_has_error(connection)) {
        fprintf(stderr, "Error: Unable to connect to X server\n");
        exit(1);
    }

    // Получаем экран
    const xcb_setup_t *setup = xcb_get_setup(connection);
    xcb_screen_iterator_t screen_iter = xcb_setup_roots_iterator(setup);
    xcb_screen_t *screen = screen_iter.data;
    if (screen == NULL) {
        fprintf(stderr, "Error: Unable to get screen\n");
        exit(1);
    }

    printf("Screen root visual: %08x\n", screen->root_visual);

    // Выводим значения констант
    printf("XCB_GC_FOREGROUND: %08x\n", XCB_GC_FOREGROUND);
    printf("XCB_GC_BACKGROUND: %08x\n", XCB_GC_BACKGROUND);
    printf("XCB_GC_GRAPHICS_EXPOSURES: %08x\n", XCB_GC_GRAPHICS_EXPOSURES);
    printf("XCB_CW_BACK_PIXEL: %08x\n", XCB_CW_BACK_PIXEL);
    printf("XCB_CW_EVENT_MASK: %08x\n", XCB_CW_EVENT_MASK);
    printf("XCB_EVENT_MASK_EXPOSURE: %08x\n", XCB_EVENT_MASK_EXPOSURE);
    printf("XCB_EVENT_MASK_BUTTON_PRESS: %08x\n", XCB_EVENT_MASK_BUTTON_PRESS);

    // Создаём окно
    xcb_window_t window = xcb_generate_id(connection);
    uint32_t values[2];
    values[0] = screen->white_pixel; // Белый фон
    values[1] = XCB_EVENT_MASK_EXPOSURE | XCB_EVENT_MASK_BUTTON_PRESS;
    xcb_create_window(
        connection,
        XCB_COPY_FROM_PARENT,
        window,
        screen->root,
        100, 100, 400, 300, 1,
        XCB_WINDOW_CLASS_INPUT_OUTPUT,
        screen->root_visual,
        XCB_CW_BACK_PIXEL | XCB_CW_EVENT_MASK,
        values
    );

    // Отображаем окно
    xcb_map_window(connection, window);
    xcb_flush(connection);
    printf("Window created and mapped\n");
    sleep(1);

    // Создаём графический контекст (GC)
    xcb_gcontext_t gc = xcb_generate_id(connection);
    uint32_t gc_values[3];
    gc_values[0] = screen->black_pixel; // Цвет переднего плана (по умолчанию)
    gc_values[1] = screen->white_pixel; // Цвет фона
    gc_values[2] = 0; // Графические экспозиции
    xcb_create_gc(
        connection,
        gc,
        window,
        XCB_GC_FOREGROUND | XCB_GC_BACKGROUND | XCB_GC_GRAPHICS_EXPOSURES,
        gc_values
    );
    printf("Graphics context created: %08x\n", gc);

    // Определяем цвета прямоугольников
    uint32_t colors[5];
    colors[0] = screen->black_pixel; // Чёрный
    colors[1] = screen->white_pixel; // Белый
    colors[2] = 0x00FF0000; // Красный
    colors[3] = 0x0000FF00; // Зелёный
    colors[4] = 0x000000FF; // Синий

    printf("Allocated colors: R=%08x, G=%08x, B=%08x\n", colors[2], colors[3], colors[4]);

    // Определяем прямоугольники
    xcb_rectangle_t rectangles[5] = {
        {50, 50, 50, 50},   // Прямоугольник 0
        {150, 50, 50, 50},  // Прямоугольник 1
        {250, 50, 50, 50},  // Прямоугольник 2
        {50, 150, 50, 50},  // Прямоугольник 3
        {150, 150, 50, 50}  // Прямоугольник 4
    };

    // Функция для рисования прямоугольников
    void draw_rectangles() {
        printf("Drawing rectangles...\n");
        for (int i = 0; i < 5; i++) {
            uint32_t mask = XCB_GC_FOREGROUND;
            uint32_t value = colors[i];
            xcb_change_gc(connection, gc, mask, &value);

            printf("Changing GC foreground to: %08x\n", value);
            printf("Filling rectangle %d at (%d, %d) with size %dx%d\n",
                   i, rectangles[i].x, rectangles[i].y, rectangles[i].width, rectangles[i].height);

            xcb_poly_fill_rectangle(connection, window, gc, 1, &rectangles[i]);
        }
        xcb_flush(connection);
    }

    // Рисуем прямоугольники
    draw_rectangles();

    // Основной цикл обработки событий
    xcb_generic_event_t *event;
    while ((event = xcb_wait_for_event(connection))) {
        switch (event->response_type & ~0x80) {
            case XCB_EXPOSE:
                printf("Expose event received, redrawing rectangles...\n");
                draw_rectangles();
                break;
            case XCB_BUTTON_PRESS:
                printf("Button press event received, exiting...\n");
                goto exit_loop;
            default:
                break;
        }
        free(event);
    }

exit_loop:
    // Освобождаем ресурсы
    xcb_free_gc(connection, gc);
    xcb_disconnect(connection);

    return 0;
}