unit xcb_sliders;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

interface

uses
  SysUtils, xcbunit, ctypes;

type
  fptr_t = procedure(b_id: Integer);
  obj_type = (DRAGGABLE, DRAGGABLE_X, DRAGGABLE_Y);

  button_t = record
    id: xcb_drawable_t;
    obj_type: obj_type;
    label_: array[0..99] of Char;
    label_orig: Char;
    label_w: Char;
    label_h: Char;
    drag_state: Integer;
    offset: array[0..1] of cuint32;
    origin: array[0..1] of cuint32;
    click_callback: fptr_t;
  end;

  Pbutton_t = ^button_t;

const
  BUTTON_HEAP_SIZE = 20;

var
  c: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  gc: xcb_gcontext_t;
  win: xcb_drawable_t;
  back_pixmap: xcb_drawable_t;
  winwidth: Integer;
  winheight: Integer;
  fill: xcb_gcontext_t;
  button_heap: array[0..BUTTON_HEAP_SIZE - 1] of button_t;
  button_heap_counter: Integer = 0;

procedure alloc_button(b: button_t);
function create_button(otype: obj_type; label_: PChar; x, y: Integer; click_callback: fptr_t): button_t;
procedure draw_button(b: Pbutton_t);
procedure create_window;
procedure event_loop;
procedure x_y(b_id: Integer);
procedure x(b_id: Integer);
procedure y(b_id: Integer);

implementation

procedure alloc_button(b: button_t);
begin
  if button_heap_counter < BUTTON_HEAP_SIZE then
  begin
    button_heap[button_heap_counter] := b;
    WriteLn(Format('registered button: %d to %d', [b.id, button_heap_counter]));
    Inc(button_heap_counter);
  end
  else
  begin
    WriteLn('can''t allocate button, out of space');
    Halt(0);
  end;
end;

function create_button(otype: obj_type; label_: PChar; x, y: Integer; click_callback: fptr_t): button_t;
var
  mask: cuint32;
  values3: array[0..2] of cuint32;
  window_id: xcb_drawable_t;
  width, height, origin: Integer;
  font: xcb_font_t;
  b: button_t;
begin
  font := xcb_generate_id(c);
  xcb_open_font(c, font, Length('7x13'), '7x13');

  // Здесь должна быть функция measure_string, но для упрощения пропустим её
  width := 50; // Примерные значения
  height := 20;
  origin := 10; // Убедимся, что origin находится в диапазоне 0..255

  window_id := xcb_generate_id(c);

  mask := XCB_CW_BACK_PIXEL or XCB_CW_BORDER_PIXEL or XCB_CW_EVENT_MASK;
  values3[0] := screen^.white_pixel;
  values3[1] := screen^.black_pixel;
  values3[2] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_BUTTON_PRESS or
                XCB_EVENT_MASK_BUTTON_RELEASE or XCB_EVENT_MASK_BUTTON_MOTION or
                XCB_EVENT_MASK_ENTER_WINDOW or XCB_EVENT_MASK_LEAVE_WINDOW or
                XCB_EVENT_MASK_KEY_PRESS;

  xcb_create_window(c, XCB_COPY_FROM_PARENT, window_id, win, x, y, width, height, 1,
                    XCB_WINDOW_CLASS_INPUT_OUTPUT, screen^.root_visual, mask, @values3[0]);

  b.id := window_id;
  b.obj_type := otype;
  StrLCopy(@b.label_, label_, Length(b.label_) - 1);
  b.label_[Length(b.label_) - 1] := #0; // Явно добавляем нулевой символ в конец строки
  b.label_orig := Chr(origin); // Убедимся, что origin находится в диапазоне 0..255
  b.label_w := Chr(width);
  b.label_h := Chr(height);
  b.drag_state := 0;
  b.offset[0] := 0;
  b.offset[1] := 0;
  b.origin[0] := x;
  b.origin[1] := y;
  b.click_callback := click_callback;

  xcb_map_window(c, window_id);
  Result := b;
end;

{
procedure draw_button(b: Pbutton_t);
var
  string_len: cuint8;
begin
  string_len := StrLen(b^.label_);
  if string_len > 0 then
  begin
    // Проверяем, что label_orig находится в допустимом диапазоне
    if (Byte(b^.label_orig) >= 0) and (Byte(b^.label_orig) <= 255) then
    begin
      // Проверяем, что gc и b^.id были корректно инициализированы
      if (gc <> 0) and (b^.id <> 0) then
      begin
        // Проверяем соединение с X-сервером
        if xcb_connection_has_error(c) = 0 then
        begin
          WriteLn(Format('xcb_image_text_8: c=%d, string_len=%d, b^.id=%d, gc=%d, Byte(b^.label_orig)=%d, b^.label_=%s',
            [PtrUInt(c), string_len, b^.id, gc, Byte(b^.label_orig), b^.label_]));
          xcb_image_text_8(c, string_len, b^.id, gc, 0, Byte(b^.label_orig), b^.label_);
        end
        else
        begin
          WriteLn('Error: XCB connection is invalid');
        end;
      end
      else
      begin
        WriteLn('Error: gc or b^.id is not initialized');
      end;
    end
    else
    begin
      WriteLn('Error: label_orig out of range');
    end;
  end;
end;
}
procedure draw_button(b: Pbutton_t);
var
  string_len: cuint8;
begin
  string_len := StrLen(b^.label_);
  if string_len > 0 then
  begin
    // Проверяем, что gc и b^.id были корректно инициализированы
    if (gc <> 0) and (b^.id <> 0) then
    begin
      // Проверяем соединение с X-сервером
      if xcb_connection_has_error(c) = 0 then
      begin
        WriteLn(Format('xcb_poly_text_8: c=%d, string_len=%d, b^.id=%d, gc=%d, Byte(b^.label_orig)=%d, b^.label_=%s',
          [PtrUInt(c), string_len, b^.id, gc, Byte(b^.label_orig), b^.label_]));
        xcb_poly_text_8(c, b^.id, gc, 0, Byte(b^.label_orig), string_len, b^.label_);
      end
      else
      begin
        WriteLn('Error: XCB connection is invalid');
      end;
    end
    else
    begin
      WriteLn('Error: gc or b^.id is not initialized');
    end;
  end;
end;


procedure create_window;
var
  mask: cuint32;
  values: array[0..2] of cuint32;
  rect: xcb_rectangle_t;
begin
  back_pixmap := xcb_generate_id(c);
  xcb_create_pixmap(c, screen^.root_depth, back_pixmap, screen^.root, 800, 800);

  fill := xcb_generate_id(c);
  mask := XCB_GC_FOREGROUND;
  values[0] := screen^.white_pixel;
  xcb_create_gc(c, fill, back_pixmap, mask, @values[0]);

  gc := xcb_generate_id(c);
  mask := XCB_GC_FOREGROUND or XCB_GC_BACKGROUND or XCB_GC_GRAPHICS_EXPOSURES;
  values[0] := screen^.black_pixel;
  values[1] := screen^.white_pixel;
  values[2] := 0;
  xcb_create_gc(c, gc, screen^.root, mask, @values[0]);

  win := xcb_generate_id(c);
  mask := XCB_CW_BACK_PIXMAP or XCB_CW_EVENT_MASK;
  values[0] := back_pixmap;
  values[1] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_BUTTON_PRESS or
               XCB_EVENT_MASK_STRUCTURE_NOTIFY or XCB_EVENT_MASK_BUTTON_RELEASE or
               XCB_EVENT_MASK_KEY_PRESS or XCB_EVENT_MASK_BUTTON_MOTION;

  xcb_create_window(c, screen^.root_depth, win, screen^.root, 0, 0, 300, 300, 0,
                    XCB_WINDOW_CLASS_INPUT_OUTPUT, screen^.root_visual, mask, @values[0]);

  xcb_map_window(c, win);

  rect.x := 0;
  rect.y := 0;
  rect.width := 800;
  rect.height := 800;
  xcb_poly_fill_rectangle(c, back_pixmap, fill, 1, @rect);
end;

procedure event_loop;
var
  e: Pxcb_generic_event_t;
  mev: Pxcb_motion_notify_event_t;
  ev: Pxcb_button_press_event_t;
  ev_release: Pxcb_button_release_event_t;
  b: Pbutton_t;
  i: Integer;
begin
  while True do
  begin
    e := xcb_wait_for_event(c);
    if e = nil then Break;

    case e^.response_type and not $80 of
      XCB_MOTION_NOTIFY:
      begin
        mev := Pxcb_motion_notify_event_t(e);
        for i := 0 to button_heap_counter - 1 do
        begin
          b := @button_heap[i];
          if (mev^.event = b^.id) and (b^.drag_state = 1) then
          begin
            WriteLn(Format('Dragging button %d: origin=(%d, %d), offset=(%d, %d), event_x=%d, event_y=%d',
              [i, b^.origin[0], b^.origin[1], b^.offset[0], b^.offset[1], mev^.event_x, mev^.event_y]));

            // Проверка значений координат
            if (b^.origin[0] >= 0) and (b^.origin[1] >= 0) and
               (b^.offset[0] >= 0) and (b^.offset[1] >= 0) and
               (mev^.event_x >= 0) and (mev^.event_y >= 0) then
            begin
WriteLn('IF');
              case b^.obj_type of
                DRAGGABLE:
                begin
WriteLn('DRAGGABLE:');
                  b^.origin[0] := b^.origin[0] + mev^.event_x - b^.offset[0];
                  b^.origin[1] := b^.origin[1] + mev^.event_y - b^.offset[1];
                  xcb_configure_window(c, b^.id, XCB_CONFIG_WINDOW_X or XCB_CONFIG_WINDOW_Y, @b^.origin);
                  b^.click_callback(i);
                end;
                DRAGGABLE_X:
                begin
WriteLn('DRAGGABLE_X:');
                  b^.origin[0] := b^.origin[0] + mev^.event_x - b^.offset[0];
                  xcb_configure_window(c, b^.id, XCB_CONFIG_WINDOW_X or XCB_CONFIG_WINDOW_Y, @b^.origin);
                  b^.click_callback(i);
                end;
                DRAGGABLE_Y:
                begin
WriteLn('DRAGGABLE_Y:');
                  b^.origin[1] := b^.origin[1] + mev^.event_y - b^.offset[1];
                  xcb_configure_window(c, b^.id, XCB_CONFIG_WINDOW_X or XCB_CONFIG_WINDOW_Y, @b^.origin);
                  b^.click_callback(i);
                end;
              end;
            end
            else
            begin
              WriteLn('Error: Invalid origin, offset, or event coordinates');
            end;
            Break;
          end;
        end;
        xcb_flush(c);
      end;
      XCB_BUTTON_PRESS:
      begin
        ev := Pxcb_button_press_event_t(e);
        for i := 0 to button_heap_counter - 1 do
        begin
          b := @button_heap[i];
          if ev^.event = b^.id then
          begin
            b^.offset[0] := ev^.event_x;
            b^.offset[1] := ev^.event_y;
            b^.drag_state := 1;
            Break;
          end;
        end;
      end;
      XCB_BUTTON_RELEASE:
      begin
        ev_release := Pxcb_button_release_event_t(e);
        for i := 0 to button_heap_counter - 1 do
        begin
          b := @button_heap[i];
          if ev_release^.event = b^.id then
          begin
            b^.drag_state := 0;
            Break;
          end;
        end;
      end;
      XCB_CONFIGURE_NOTIFY:
      begin
        if Pxcb_configure_notify_event_t(e)^.window = win then
        begin
          winwidth := Pxcb_configure_notify_event_t(e)^.width;
          winheight := Pxcb_configure_notify_event_t(e)^.height;
        end;
      end;
      XCB_EXPOSE:
      begin
        for i := 0 to button_heap_counter - 1 do
        begin
          b := @button_heap[i];
          if Pxcb_expose_event_t(e)^.window = b^.id then
          begin
            draw_button(b);
          end;
        end;
        xcb_flush(c);
      end;
      XCB_KEY_PRESS:
      begin
        if Pxcb_key_press_event_t(e)^.detail = 9 then
          Exit;
      end;
    end;
//    FreeMem(e);
  end;
end;

procedure x_y(b_id: Integer);
var
  b: Pbutton_t;
  val, val2: Integer;
  str: array[0..14] of Char;
  rect: xcb_rectangle_t;
begin
  b := @button_heap[b_id];
  val := (b^.origin[0] * 100) div winwidth;
  val2 := (b^.origin[1] * 100) div winheight;
  StrFmt(str, 'xy:%d,%d', [val, val2]);

  rect.x := 20;
  rect.y := 0;
  rect.width := 80;
  rect.height := 40;
  xcb_poly_fill_rectangle(c, win, fill, 1, @rect);

  xcb_image_text_8(c, StrLen(str), win, gc, 20, 20, str);
end;

procedure x(b_id: Integer);
var
  b: Pbutton_t;
  val: Integer;
  str: array[0..14] of Char;
  rect: xcb_rectangle_t;
begin
  b := @button_heap[b_id];
  val := (b^.origin[0] * 100) div winwidth;
  StrFmt(str, 'x:%d', [val]);

  rect.x := 100;
  rect.y := 0;
  rect.width := 80;
  rect.height := 40;
  xcb_poly_fill_rectangle(c, win, fill, 1, @rect);

  xcb_image_text_8(c, StrLen(str), win, gc, 100, 20, str);
end;

procedure y(b_id: Integer);
var
  b: Pbutton_t;
  val: Integer;
  str: array[0..14] of Char;
  rect: xcb_rectangle_t;
begin
  b := @button_heap[b_id];
  val := (b^.origin[1] * 100) div winheight;
  StrFmt(str, 'y:%d', [val]);

  rect.x := 180;
  rect.y := 0;
  rect.width := 80;
  rect.height := 40;
  xcb_poly_fill_rectangle(c, win, fill, 1, @rect);

  xcb_image_text_8(c, StrLen(str), win, gc, 180, 20, str);
end;

end.