unit xcbwin;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

interface

uses
  SysUtils, xcbunit, ctypes;

type
  TXcbwin = class
  private
    FConnection: Pxcb_connection_t;
    FScreen: Pxcb_screen_t;
    FWindow: xcb_window_t;
    FClosed: Boolean;
    FWidth: cuint16;
    FHeight: cuint16;
    FVerbose: Boolean;
    FColormap: xcb_colormap_t;
    FContext: xcb_gcontext_t;
    FContextCurrent: xcb_gcontext_t;
    FLastColor: cuint32;
    procedure SetColor(AColor: cuint32);
    procedure CheckForEvent;
    function GenerateContext(AColor: cuint32): xcb_gcontext_t;
  public
    constructor Create(AVerbose: Boolean = False);
    destructor Destroy; override;
    procedure Open(AWidth, AHeight: cuint16);
    procedure DrawPoint(AX, AY: cuint16);
    procedure DrawLine(AX0, AY0, AX1, AY1: cuint16);
    procedure DrawRectangle(AX, AY, AWidth, AHeight: cuint16);
    procedure DrawFilledRectangle(AX, AY, AWidth, AHeight: cuint16);
    procedure DrawCircle(AX, AY, AWidth, AHeight: cuint16);
    procedure DrawFilledCircle(AX, AY, AWidth, AHeight: cuint16);
    procedure DrawText(AX, AY: cuint16; const AText: string);
    procedure SetColorRGB(AR, AG, AB: cuint8);
    procedure Wait;
    procedure Flush;
    procedure Clear;
    procedure CloseWindow; // Переименовано с Close на CloseWindow
  end;

implementation

constructor TXcbwin.Create(AVerbose: Boolean);
begin
  FVerbose := AVerbose;
  FClosed := True;
  FConnection := xcb_connect(nil, nil);
  if xcb_connection_has_error(FConnection) <> 0 then
  begin
    WriteLn('Cannot open display');
    Halt(1);
  end;
  FScreen := xcb_setup_roots_iterator(xcb_get_setup(FConnection)).data;
  if FVerbose then
  begin
    WriteLn('Screen Information');
    WriteLn('Width: ', FScreen^.width_in_pixels);
    WriteLn('Height: ', FScreen^.height_in_pixels);
  end;
end;

destructor TXcbwin.Destroy;
begin
  if not FClosed then
    CloseWindow; // Закрываем окно, если оно ещё не закрыто
  inherited Destroy;
end;

procedure TXcbwin.CloseWindow;
begin
  if FConnection <> nil then
  begin
    xcb_disconnect(FConnection);
    FConnection := nil; // Обнуляем указатель, чтобы избежать повторного освобождения
    FClosed := True;
  end;
end;

procedure TXcbwin.Open(AWidth, AHeight: cuint16);
var
  Mask: cuint32;
  Values: array[0..3] of cuint32;
  Cookie: xcb_void_cookie_t;
  Error: Pxcb_generic_error_t;
begin
  FWidth := AWidth;
  FHeight := AHeight;
  FWindow := xcb_generate_id(FConnection);
  Mask := XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK;
  Values[0] := FScreen^.white_pixel;
  Values[1] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_KEY_PRESS;
  Cookie := xcb_create_window(FConnection, XCB_COPY_FROM_PARENT, FWindow, FScreen^.root,
                    0, 0, FWidth, FHeight, 10, XCB_WINDOW_CLASS_INPUT_OUTPUT,
                    FScreen^.root_visual, Mask, @Values[0]);
  Error := xcb_request_check(FConnection, Cookie);
  if Error <> nil then
  begin
    WriteLn('Error: Failed to create window');
    FreeMem(Error);
  end;

  FColormap := xcb_generate_id(FConnection);
  Cookie := xcb_create_colormap(FConnection, XCB_COLORMAP_ALLOC_NONE, FColormap, FWindow, FScreen^.root_visual);
  Error := xcb_request_check(FConnection, Cookie);
  if Error <> nil then
  begin
    WriteLn('Error: Failed to create colormap');
    FreeMem(Error);
  end;

  Cookie := xcb_map_window(FConnection, FWindow);
  Error := xcb_request_check(FConnection, Cookie);
  if Error <> nil then
  begin
    WriteLn('Error: Failed to map window');
    FreeMem(Error);
  end;

  FClosed := False;

  // Создаём графический контекст для отрисовки
  FContextCurrent := GenerateContext(FScreen^.white_pixel);
  WriteLn('Graphics context created: ', FContextCurrent);

  // Очищаем окно белым цветом
  Clear;
end;

procedure TXcbwin.DrawPoint(AX, AY: cuint16);
var
  Point: xcb_point_t;
begin
  Point.x := AX;
  Point.y := AY;
  xcb_poly_point(FConnection, XCB_COORD_MODE_ORIGIN, FWindow, FContextCurrent, 1, @Point);
  Flush;
end;

procedure TXcbwin.DrawLine(AX0, AY0, AX1, AY1: cuint16);
var
  Points: array[0..1] of xcb_point_t;
begin
  Points[0].x := AX0;
  Points[0].y := AY0;
  Points[1].x := AX1;
  Points[1].y := AY1;
  xcb_poly_line(FConnection, XCB_COORD_MODE_ORIGIN, FWindow, FContextCurrent, 2, @Points[0]);
  Flush;
end;

procedure TXcbwin.DrawRectangle(AX, AY, AWidth, AHeight: cuint16);
var
  Rect: xcb_rectangle_t;
begin
  Rect.x := AX;
  Rect.y := AY;
  Rect.width := AWidth;
  Rect.height := AHeight;
  xcb_poly_rectangle(FConnection, FWindow, FContextCurrent, 1, @Rect);
  Flush;
end;

procedure TXcbwin.DrawFilledRectangle(AX, AY, AWidth, AHeight: cuint16);
var
  Rect: xcb_rectangle_t;
  Cookie: xcb_void_cookie_t;
  Error: Pxcb_generic_error_t;
begin
  Rect.x := AX;
  Rect.y := AY;
  Rect.width := AWidth;
  Rect.height := AHeight;
  WriteLn('Drawing rectangle: X=', AX, ', Y=', AY, ', Width=', AWidth, ', Height=', AHeight);
  Cookie := xcb_poly_fill_rectangle(FConnection, FWindow, FContextCurrent, 1, @Rect);
  Error := xcb_request_check(FConnection, Cookie);
  if Error <> nil then
  begin
    WriteLn('Error: Failed to draw filled rectangle');
    FreeMem(Error);
  end;
  Flush;
end;

procedure TXcbwin.DrawCircle(AX, AY, AWidth, AHeight: cuint16);
var
  Arc: xcb_arc_t;
begin
  Arc.x := AX;
  Arc.y := AY;
  Arc.width := AWidth;
  Arc.height := AHeight;
  Arc.angle1 := 0;
  Arc.angle2 := 360 shl 6;
  xcb_poly_arc(FConnection, FWindow, FContextCurrent, 1, @Arc);
  Flush;
end;

procedure TXcbwin.DrawFilledCircle(AX, AY, AWidth, AHeight: cuint16);
var
  Arc: xcb_arc_t;
begin
  Arc.x := AX;
  Arc.y := AY;
  Arc.width := AWidth;
  Arc.height := AHeight;
  Arc.angle1 := 0;
  Arc.angle2 := 360 shl 6;
  xcb_poly_fill_arc(FConnection, FWindow, FContextCurrent, 1, @Arc);
  Flush;
end;

procedure TXcbwin.DrawText(AX, AY: cuint16; const AText: string);
begin
  xcb_image_text_8(FConnection, Length(AText), FWindow, FContextCurrent, AX, AY, PChar(AText));
  Flush;
end;

procedure TXcbwin.SetColorRGB(AR, AG, AB: cuint8);
var
  Color: cuint32;
begin
  Color := (AR shl 16) or (AG shl 8) or AB;
  WriteLn('Setting color: R=', AR, ', G=', AG, ', B=', AB, ', Color=', IntToHex(Color, 8));
  SetColor(Color);
end;

procedure TXcbwin.SetColor(AColor: cuint32);
var
  Cookie: xcb_void_cookie_t;
  Error: Pxcb_generic_error_t;
begin
  if AColor <> FLastColor then
  begin
    WriteLn('Changing foreground color to: ', IntToHex(AColor, 8));
    Cookie := xcb_change_gc(FConnection, FContextCurrent, XCB_GC_FOREGROUND, @AColor);
    Error := xcb_request_check(FConnection, Cookie);
    if Error <> nil then
    begin
      WriteLn('Error: Failed to change GC foreground');
      FreeMem(Error);
    end;
    FLastColor := AColor;
  end;
end;

procedure TXcbwin.Wait;
var
  Event: Pxcb_generic_event_t;
begin
  Flush;
  repeat
    Event := xcb_wait_for_event(FConnection);
    if Event <> nil then
    begin
      case Event^.response_type and not $80 of
        XCB_EXPOSE:
          Flush;
        XCB_KEY_PRESS:
          Break;
      end;
//      FreeMem(Event);
    end;
  until False;
end;

procedure TXcbwin.Flush;
begin
  WriteLn('Flushing the connection...');
  if xcb_flush(FConnection) <= 0 then
  begin
    WriteLn('Error: Failed to flush the connection');
  end;
end;	

procedure TXcbwin.Clear;
begin
  WriteLn('Clearing the window with white color...');
  SetColor(FScreen^.white_pixel);
  DrawFilledRectangle(0, 0, FWidth, FHeight);
  SetColor(0); // Устанавливаем чёрный цвет для последующей отрисовки
  Flush;
end;

function TXcbwin.GenerateContext(AColor: cuint32): xcb_gcontext_t;
var
  Mask: cuint32;
  Values: array[0..3] of cuint32;
  Cookie: xcb_void_cookie_t;
  Error: Pxcb_generic_error_t;
begin
  Result := xcb_generate_id(FConnection);
  Mask := XCB_GC_FOREGROUND or XCB_GC_BACKGROUND or XCB_GC_GRAPHICS_EXPOSURES;
  Values[0] := AColor;
  Values[1] := FScreen^.white_pixel;
  Values[2] := 1;
  Values[3] := 0;
  WriteLn('Creating graphics context: Color=', IntToHex(AColor, 8));
  Cookie := xcb_create_gc(FConnection, Result, FWindow, Mask, @Values[0]);
  Error := xcb_request_check(FConnection, Cookie);
  if Error <> nil then
  begin
    WriteLn('Error: Failed to create graphics context');
    FreeMem(Error);
  end;
end;

procedure TXcbwin.CheckForEvent;
var
  Event: Pxcb_generic_event_t;
begin
  Event := xcb_poll_for_event(FConnection);
  if Event <> nil then
  begin
    if (Event^.response_type and not $80) = XCB_EXPOSE then
      Flush;
    FreeMem(Event);
  end;
end;

end.