program demo_advanced;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_const, xcb_bindings, xproto_bindings, xcb_events_bindings, BaseUnix, SysUtils;

//  procedure cfree(event: Pointer); cdecl; external 'c' name 'free'; // Use standard C free

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: xcb_window_t;
  event: Pxcb_generic_event_t;
  has_error: cint;
  running: Boolean = True;
  event_count: integer = 0;
  max_events: integer = 50;

procedure SignalHandler(sig: cint); cdecl;
begin
  if sig = SIGINT then
  begin
    WriteLn;
    WriteLn('Received Ctrl+C, shutting down...');
    running := False;
  end;
end;

{ Print names of modifiers present in mask }
procedure PrintModifiers(mask: uint32_t);
const
  MODIFIERS: array[0..12] of string = (
    'Shift', 'Lock', 'Ctrl', 'Alt',
    'Mod2', 'Mod3', 'Mod4', 'Mod5',
    'Button1', 'Button2', 'Button3', 'Button4', 'Button5'
  );
var
  i: integer;
  first: Boolean = True;
begin
  Write('Modifier mask: ');
  for i := 0 to High(MODIFIERS) do
  begin
    if (mask and (1 shl i)) <> 0 then
    begin
      if not first then
        Write(', ');
      Write(MODIFIERS[i]);
      first := False;
    end;
  end;
  WriteLn;
end;

procedure CreateWindow;
var
  setup: Pxcb_setup_t;
  screen_iterator: xcb_screen_iterator_t;
  mask: uint32_t;
  values: array[0..1] of uint32_t;
begin
  // Get setup and screen
  setup := xcb_get_setup(conn);
  screen_iterator := xcb_setup_roots_iterator(setup);
  screen := screen_iterator.data;
  
  // Create window
  window := xcb_generate_id(conn);
  
  // Set window attributes
  mask := XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK;
  values[0] := screen^.white_pixel;
  values[1] := XCB_EVENT_MASK_EXPOSURE       or XCB_EVENT_MASK_BUTTON_PRESS   or
                XCB_EVENT_MASK_BUTTON_RELEASE or XCB_EVENT_MASK_POINTER_MOTION or
                XCB_EVENT_MASK_ENTER_WINDOW   or XCB_EVENT_MASK_LEAVE_WINDOW   or
                XCB_EVENT_MASK_KEY_PRESS      or XCB_EVENT_MASK_KEY_RELEASE;
  
  xcb_create_window(conn,
                    0,                             // depth
                    window,
                    screen^.root,                  // parent window
                    0, 0,                          // x, y
                    150, 150,                      // width, height
                    10,                            // border_width
                    XCB_WINDOW_CLASS_INPUT_OUTPUT, // class
                    screen^.root_visual,           // visual
                    mask, @values);                // masks

  // Map the window on the screen
  xcb_map_window(conn, window);
  xcb_flush(conn);
  
  WriteLn('Created window with ID: ', window);
  WriteLn('Window size: 150x150 at position 0,0');
end;

procedure HandleEvent;
var
  event_type: uint8_t;
begin
  event_type := event^.response_type and not $80;
  
  case event_type of
    XCB_EXPOSE: 
    begin
      with Pxcb_expose_event_t(event)^ do
      begin
        WriteLn('Window ', window, ' exposed. Region to be redrawn at location (', 
                x, ',', y, '), with dimension (', width, ',', height, ')');
      end;
    end;
    
    XCB_BUTTON_PRESS: 
    begin
      with Pxcb_button_press_event_t(event)^ do
      begin
        PrintModifiers(state);
        
        case detail of
          4: WriteLn('Wheel Button up in window ', event, ', at coordinates (', 
                     event_x, ',', event_y, ')');
          5: WriteLn('Wheel Button down in window ', event, ', at coordinates (', 
                     event_x, ',', event_y, ')');
          else WriteLn('Button ', detail, ' pressed in window ', event, 
                      ', at coordinates (', event_x, ',', event_y, ')');
        end;
      end;
    end;
    
    XCB_BUTTON_RELEASE: 
    begin
      with Pxcb_button_release_event_t(event)^ do
      begin
        PrintModifiers(state);
        WriteLn('Button ', detail, ' released in window ', event, 
                ', at coordinates (', event_x, ',', event_y, ')');
      end;
    end;
    
    XCB_MOTION_NOTIFY: 
    begin
      with Pxcb_motion_notify_event_t(event)^ do
      begin
        WriteLn('Mouse moved in window ', event, ', at coordinates (', 
                event_x, ',', event_y, ')');
      end;
    end;
    
    XCB_ENTER_NOTIFY: 
    begin
      with Pxcb_enter_notify_event_t(event)^ do
      begin
        WriteLn('Mouse entered window ', event, ', at coordinates (', 
                event_x, ',', event_y, ')');
      end;
    end;
    
    XCB_LEAVE_NOTIFY: 
    begin
      with Pxcb_leave_notify_event_t(event)^ do
      begin
        WriteLn('Mouse left window ', event, ', at coordinates (', 
                event_x, ',', event_y, ')');
      end;
    end;
    
    XCB_KEY_PRESS: 
    begin
      with Pxcb_key_press_event_t(event)^ do
      begin
        PrintModifiers(state);
        WriteLn('Key pressed in window ', event);
      end;
    end;
    
    XCB_KEY_RELEASE: 
    begin
      with Pxcb_key_release_event_t(event)^ do
      begin
        PrintModifiers(state);
        WriteLn('Key released in window ', event);
      end;
    end;
    
    else
      WriteLn('Unknown event: ', event_type);
  end;
end;

begin
  WriteLn('XCB Advanced Demo (Stable Version)');
  WriteLn('==================================');
  WriteLn('Based on C example from XCB documentation');
  WriteLn('Will process up to ', max_events, ' events');
  WriteLn('Uses safe approach: no event freeing');
  WriteLn;
  
  // Setup signal handler
  FpSignal(SIGINT, @SignalHandler);
  
  // Open the connection to the X server
  conn := xcb_connect(nil, nil);
  
  if conn = nil then
  begin
    WriteLn('ERROR: Failed to connect to X server');
    Halt(1);
  end;
  
  has_error := xcb_connection_has_error(conn);
  if has_error <> 0 then
  begin
    WriteLn('ERROR: Connection error: ', has_error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  
  WriteLn('Connected to X server');
  
  // Create the window
  CreateWindow;

  WriteLn('Event loop started. Interact with the window or press Ctrl+C to exit');
  WriteLn;
  
  // Main event loop - НЕ освобождаем события (безопасный подход)
  while running and (event_count < max_events) do
  begin
    event := xcb_poll_for_event(conn);
    
    if event <> nil then
    begin
      Inc(event_count);
      HandleEvent;

      // НЕ освобождаем события - это безопасно для демо
      // FreeMem(event); // ЗАКОММЕНТИРОВАНО для безопасности
      CFree(event);
    end
    else
    begin
      // No events, small delay
      Sleep(10);
    end;
  end;
  
  if event_count >= max_events then
    WriteLn('Reached event limit (', max_events, ')');
  
  WriteLn('Processed ', event_count, ' events');
  WriteLn('Destroying window...');
  
  // Cleanup
  xcb_destroy_window(conn, window);
  xcb_flush(conn);
  xcb_disconnect(conn);
  
  WriteLn('Disconnected from X server');
  WriteLn('Demo finished successfully!');
end.