program demo_advanced_wait;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, xproto_bindings, xcb_events_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: xcb_window_t;
  event: Pxcb_generic_event_t;
  has_error: cint;
  running: Boolean = True;

procedure SignalHandler(sig: cint); cdecl;
begin
  if sig = SIGINT then
  begin
    WriteLn;
    WriteLn('Received Ctrl+C, shutting down...');
    running := False;
  end;
end;

procedure CreateWindow;
var
  setup: Pxcb_setup_t;
  screen_iterator: xcb_screen_iterator_t;
  mask: uint32_t;
  values: array[0..1] of uint32_t;
begin
  setup := xcb_get_setup(conn);
  screen_iterator := xcb_setup_roots_iterator(setup);
  screen := screen_iterator.data;
  
  window := xcb_generate_id(conn);
  
  mask := XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK;
  values[0] := screen^.white_pixel;
  values[1] := XCB_EVENT_MASK_EXPOSURE       or XCB_EVENT_MASK_BUTTON_PRESS   or
                XCB_EVENT_MASK_BUTTON_RELEASE or XCB_EVENT_MASK_POINTER_MOTION or
                XCB_EVENT_MASK_KEY_PRESS      or XCB_EVENT_MASK_KEY_RELEASE;
  
  xcb_create_window(conn, 0, window, screen^.root,
                    100, 100, 200, 200, 10,
                    XCB_WINDOW_CLASS_INPUT_OUTPUT,
                    screen^.root_visual, mask, @values);

  xcb_map_window(conn, window);
  xcb_flush(conn);
  
  WriteLn('Created window ', window, ' at 100,100 (200x200)');
end;

procedure HandleEvent;
var
  event_type: uint8_t;
begin
  event_type := event^.response_type and not $80;
  
  case event_type of
    XCB_EXPOSE: 
      WriteLn('>>> Window exposed');
    
    XCB_BUTTON_PRESS: 
      with Pxcb_button_press_event_t(event)^ do
        case detail of
          1: WriteLn('>>> Left mouse button pressed at (', event_x, ',', event_y, ')');
          2: WriteLn('>>> Middle mouse button pressed at (', event_x, ',', event_y, ')');
          3: WriteLn('>>> Right mouse button pressed at (', event_x, ',', event_y, ')');
          4: WriteLn('>>> Mouse wheel up');
          5: WriteLn('>>> Mouse wheel down');
        end;
    
    XCB_BUTTON_RELEASE: 
      WriteLn('>>> Mouse button released');
    
    XCB_MOTION_NOTIFY: 
      with Pxcb_motion_notify_event_t(event)^ do
        WriteLn('>>> Mouse motion to (', event_x, ',', event_y, ')');
    
    XCB_KEY_PRESS: 
      WriteLn('>>> Key pressed');
    
    XCB_KEY_RELEASE: 
      WriteLn('>>> Key released');
    
    else
      WriteLn('>>> Other event: ', event_type);
  end;
end;

begin
  WriteLn('XCB Advanced Demo (Blocking Wait Version)');
  WriteLn('==========================================');
  WriteLn('Uses xcb_wait_for_event for reliable operation');
  WriteLn('Press Ctrl+C to exit');
  WriteLn;
  
  FpSignal(SIGINT, @SignalHandler);
  
  conn := xcb_connect(nil, nil);
  if conn = nil then
  begin
    WriteLn('Failed to connect to X server');
    Halt(1);
  end;
  
  has_error := xcb_connection_has_error(conn);
  if has_error <> 0 then
  begin
    WriteLn('Connection error: ', has_error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  
  WriteLn('Connected to X server');
  CreateWindow;
  WriteLn('Waiting for events...');
  
  // Используем xcb_wait_for_event который безопасен для FreeMem
  while running do
  begin
    event := xcb_wait_for_event(conn);
    
    if event <> nil then
    begin
      HandleEvent;
      FreeMem(event); // Безопасно с xcb_wait_for_event
    end
    else
    begin
      WriteLn('Error: null event received');
      running := False;
    end;
  end;
  
  WriteLn('Cleaning up...');
  xcb_destroy_window(conn, window);
  xcb_disconnect(conn);
  WriteLn('Demo completed');
end.