program demo_complete;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, xproto_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: xcb_window_t;
  event: Pxcb_generic_event_t;
  setup: Pxcb_setup_t;
  has_error: cint;

procedure DisplaySystemInfo;
var
  screen_iterator: xcb_screen_iterator_t;
begin
  setup := xcb_get_setup(conn);
  screen_iterator := xcb_setup_roots_iterator(setup);
  screen := screen_iterator.data;
  
  WriteLn('XCB System Information');
  WriteLn('======================');
  WriteLn('Protocol: ', setup^.protocol_major_version, '.', setup^.protocol_minor_version);
  WriteLn('Release: ', setup^.release_number);
  WriteLn('Resource ID Base: ', setup^.resource_id_base);
  WriteLn('Screens: ', setup^.roots_len);
  WriteLn('Screen Size: ', screen^.width_in_pixels, 'x', screen^.height_in_pixels);
  WriteLn('File Descriptor: ', xcb_get_file_descriptor(conn));
  WriteLn;
end;

procedure CreateDemoWindow;
var
  value_mask: uint32_t;
  value_list: array[0..0] of uint32_t;
begin
  // Create window
  window := xcb_generate_id(conn);
  
  // Set event mask
  value_mask := XCB_CW_EVENT_MASK;
  value_list[0] := XCB_EVENT_MASK_EXPOSURE or 
                   XCB_EVENT_MASK_KEY_PRESS or 
                   XCB_EVENT_MASK_BUTTON_PRESS or
                   XCB_EVENT_MASK_POINTER_MOTION;
  
  xcb_create_window(conn,
    XCB_COPY_FROM_PARENT,
    window,
    screen^.root,
    150, 150, 500, 400,  // x, y, width, height
    5,                   // border width
    XCB_WINDOW_CLASS_INPUT_OUTPUT,
    screen^.root_visual,
    value_mask, @value_list);
  
  // Map window
  xcb_map_window(conn, window);
  
  // Flush requests
  xcb_flush(conn);
  
  WriteLn('Demo Window Created!');
  WriteLn('Window ID: ', window);
  WriteLn('Press any key in the window or move mouse to test events');
  WriteLn('Press Ctrl+C to exit');
  WriteLn;
end;

begin
  WriteLn('XCB Complete Demo');
  WriteLn('=================');
  WriteLn;
  
  // Connect to X server
  conn := xcb_connect(nil, nil);
  
  if conn = nil then
  begin
    WriteLn('Failed to connect to X server');
    Halt(1);
  end;
  
  has_error := xcb_connection_has_error(conn);
  if has_error <> 0 then
  begin
    WriteLn('Connection error: ', has_error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  
  WriteLn('Successfully connected to X server');
  WriteLn;
  
  // Display system information
  DisplaySystemInfo;
  
  // Create demo window
  CreateDemoWindow;
  
  // Simple event loop
  WriteLn('Starting event loop...');
  
  while True do
  begin
    event := xcb_poll_for_event(conn);
    
    if event <> nil then
    begin
      case (event^.response_type and $7F) of
        2: WriteLn('>>> Key Pressed');
        4: WriteLn('>>> Mouse Button Pressed');
        6: WriteLn('>>> Mouse Moved');
        12: WriteLn('>>> Window Exposed');
      end;
      
      // Don't free events in this simple demo
    end
    else
    begin
      Sleep(50);
    end;
  end;
  
  // Note: This demo doesn't include cleanup since it runs indefinitely
  // In a real application, you would add proper cleanup
end.