program demo_detailed_events;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, xproto_bindings, xcb_events_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: xcb_window_t;

procedure PrintEventDetails(event: Pxcb_generic_event_t);
var
  event_type: uint8_t;
begin
  event_type := event^.response_type and not $80;
  
  Write('Event: ');
  case event_type of
    XCB_EXPOSE: 
      begin
        with Pxcb_expose_event_t(event)^ do
        begin
          WriteLn('EXPOSE - Window: ', window, ' Area: (', x, ',', y, ')-[', width, 'x', height, ']');
        end;
      end;
      
    XCB_BUTTON_PRESS: 
      begin
        with Pxcb_button_press_event_t(event)^ do
        begin
          Write('BUTTON_PRESS - Button: ', detail);
          Write(' Window: ', event);
          Write(' Coordinates: (', event_x, ',', event_y, ')');
          Write(' Root: (', root_x, ',', root_y, ')');
          WriteLn(' Time: ', time);
        end;
      end;
      
    XCB_BUTTON_RELEASE: 
      begin
        with Pxcb_button_release_event_t(event)^ do
        begin
          Write('BUTTON_RELEASE - Button: ', detail);
          Write(' Window: ', event);
          Write(' Coordinates: (', event_x, ',', event_y, ')');
          WriteLn(' Time: ', time);
        end;
      end;
      
    XCB_MOTION_NOTIFY: 
      begin
        with Pxcb_motion_notify_event_t(event)^ do
        begin
          Write('MOTION_NOTIFY');
          Write(' Window: ', event);
          Write(' Coordinates: (', event_x, ',', event_y, ')');
          Write(' Root: (', root_x, ',', root_y, ')');
          WriteLn(' Time: ', time);
        end;
      end;
      
    XCB_KEY_PRESS: 
      begin
        with Pxcb_key_press_event_t(event)^ do
        begin
          Write('KEY_PRESS - Keycode: ', detail);
          Write(' Window: ', event);
          Write(' Coordinates: (', event_x, ',', event_y, ')');
          WriteLn(' Time: ', time);
        end;
      end;
      
    XCB_KEY_RELEASE: 
      begin
        with Pxcb_key_release_event_t(event)^ do
        begin
          Write('KEY_RELEASE - Keycode: ', detail);
          Write(' Window: ', event);
          Write(' Coordinates: (', event_x, ',', event_y, ')');
          WriteLn(' Time: ', time);
        end;
      end;
      
    else
      WriteLn('UNKNOWN (Type: ', event_type, ')');
  end;
end;

procedure RunDemo;
var
  event: Pxcb_generic_event_t;
  setup: Pxcb_setup_t;
  screen_iterator: xcb_screen_iterator_t;
  mask: uint32_t;
  values: array[0..1] of uint32_t;
  event_count: integer = 0;
  max_events: integer = 50;
begin
  WriteLn('XCB Detailed Events Demo');
  WriteLn('========================');
  WriteLn('Will show detailed information for ', max_events, ' events');
  WriteLn;
  
  // Connect to X server
  conn := xcb_connect(nil, nil);
  if conn = nil then
  begin
    WriteLn('Failed to connect to X server');
    Exit;
  end;
  
  if xcb_connection_has_error(conn) <> 0 then
  begin
    WriteLn('Connection error');
    xcb_disconnect(conn);
    Exit;
  end;
  
  // Get screen and create window
  setup := xcb_get_setup(conn);
  screen_iterator := xcb_setup_roots_iterator(setup);
  screen := screen_iterator.data;
  
  window := xcb_generate_id(conn);
  mask := XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK;
  values[0] := screen^.white_pixel;
  values[1] := XCB_EVENT_MASK_EXPOSURE       or XCB_EVENT_MASK_BUTTON_PRESS   or
                XCB_EVENT_MASK_BUTTON_RELEASE or XCB_EVENT_MASK_POINTER_MOTION or
                XCB_EVENT_MASK_KEY_PRESS      or XCB_EVENT_MASK_KEY_RELEASE;
  
  xcb_create_window(conn, 0, window, screen^.root,
                    200, 200, 300, 250, 10,
                    XCB_WINDOW_CLASS_INPUT_OUTPUT,
                    screen^.root_visual, mask, @values);
  
  xcb_map_window(conn, window);
  xcb_flush(conn);
  
  WriteLn('Window created. Interact with it to generate events...');
  WriteLn;
  
  // Event loop
  while event_count < max_events do
  begin
    event := xcb_poll_for_event(conn);
    
    if event <> nil then
    begin
      Inc(event_count);
      Write('Event ', event_count:2, ': ');
      PrintEventDetails(event);
      // Don't free events in this demo
    end
    else
    begin
      Sleep(50);
    end;
  end;
  
  WriteLn;
  WriteLn('Reached event limit. Cleaning up...');
  xcb_destroy_window(conn, window);
  xcb_disconnect(conn);
  WriteLn('Demo completed');
end;

begin
  RunDemo;
end.