program demo_focused;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, xproto_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: xcb_window_t;
  event: Pxcb_generic_event_t;
  event_count: integer = 0;
  max_events: integer = 30;

procedure CreateWindow;
var
  setup: Pxcb_setup_t;
  screen_iterator: xcb_screen_iterator_t;
  mask: uint32_t;
  values: array[0..1] of uint32_t;
begin
  setup := xcb_get_setup(conn);
  screen_iterator := xcb_setup_roots_iterator(setup);
  screen := screen_iterator.data;
  
  window := xcb_generate_id(conn);
  mask := XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK;
  values[0] := screen^.white_pixel;
  values[1] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_BUTTON_PRESS or 
                XCB_EVENT_MASK_KEY_PRESS or XCB_EVENT_MASK_POINTER_MOTION;
  
  xcb_create_window(conn, 0, window, screen^.root,
                    200, 200, 300, 200, 10,
                    XCB_WINDOW_CLASS_INPUT_OUTPUT,
                    screen^.root_visual, mask, @values);

  xcb_map_window(conn, window);
  xcb_flush(conn);
end;

procedure HandleEvent;
var
  event_type: uint8_t;
begin
  event_type := event^.response_type and not $80;
  
  case event_type of
    XCB_EXPOSE: 
      WriteLn('📋 Window exposed');
    
    XCB_BUTTON_PRESS: 
      with Pxcb_button_press_event_t(event)^ do
        case detail of
          1: WriteLn('🖱️  Left click at (', event_x, ',', event_y, ')');
          2: WriteLn('🖱️  Middle click at (', event_x, ',', event_y, ')');
          3: WriteLn('🖱️  Right click at (', event_x, ',', event_y, ')');
          4: WriteLn('🖱️  Mouse wheel up');
          5: WriteLn('🖱️  Mouse wheel down');
        end;
    
    XCB_KEY_PRESS: 
      WriteLn('⌨️  Key pressed');
    
    XCB_MOTION_NOTIFY: 
      with Pxcb_motion_notify_event_t(event)^ do
        if event_count mod 5 = 0 then  // Show every 5th motion event
          WriteLn('🎯 Mouse at (', event_x, ',', event_y, ')');
    
    else
      // Ignore other events
  end;
end;

begin
  WriteLn('XCB Focused Demo');
  WriteLn('================');
  WriteLn('Shows only key interactions (clicks, keys, occasional motion)');
  WriteLn('Safe memory handling - no event freeing');
  WriteLn;
  
  conn := xcb_connect(nil, nil);
  if (conn = nil) or (xcb_connection_has_error(conn) <> 0) then
  begin
    WriteLn('Failed to connect to X server');
    Halt(1);
  end;
  
  WriteLn('✅ Connected to X server');
  CreateWindow;
  WriteLn('✅ Window created at 200,200 (300x200)');
  WriteLn('👉 Click in the window or press keys');
  WriteLn('⏹️  Will stop after ', max_events, ' events');
  WriteLn;
  
  while event_count < max_events do
  begin
    event := xcb_poll_for_event(conn);
    
    if event <> nil then
    begin
      Inc(event_count);
      HandleEvent;
      // НЕ освобождаем события
    end
    else
    begin
      Sleep(50);
    end;
  end;
  
  WriteLn;
  WriteLn('✅ Reached ', max_events, ' events');
  WriteLn('🧹 Cleaning up...');
  xcb_destroy_window(conn, window);
  xcb_disconnect(conn);
  WriteLn('✅ Disconnected from X server');
  WriteLn('🎉 Demo completed successfully!');
end.