program demo_simple_working;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, xproto_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: xcb_window_t;
  event: Pxcb_generic_event_t;
  event_count: integer = 0;

procedure CreateWindow;
var
  setup: Pxcb_setup_t;
  screen_iterator: xcb_screen_iterator_t;
  mask: uint32_t;
  values: array[0..1] of uint32_t;
begin
  setup := xcb_get_setup(conn);
  screen_iterator := xcb_setup_roots_iterator(setup);
  screen := screen_iterator.data;
  
  window := xcb_generate_id(conn);
  mask := XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK;
  values[0] := screen^.white_pixel;
  values[1] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_BUTTON_PRESS or 
                XCB_EVENT_MASK_KEY_PRESS;
  
  xcb_create_window(conn, 0, window, screen^.root,
                    150, 150, 250, 250, 5,
                    XCB_WINDOW_CLASS_INPUT_OUTPUT,
                    screen^.root_visual, mask, @values);

  xcb_map_window(conn, window);
  xcb_flush(conn);
end;

function GetEventName(event_type: uint8_t): string;
begin
  case event_type of
    XCB_EXPOSE: Result := 'Expose';
    XCB_BUTTON_PRESS: Result := 'ButtonPress';
    XCB_KEY_PRESS: Result := 'KeyPress';
    else Result := 'Other(' + IntToStr(event_type) + ')';
  end;
end;

begin
  WriteLn('XCB Simple Working Demo');
  WriteLn('=======================');
  WriteLn('Safe event handling without memory issues');
  WriteLn;
  
  conn := xcb_connect(nil, nil);
  if (conn = nil) or (xcb_connection_has_error(conn) <> 0) then
  begin
    WriteLn('Failed to connect to X server');
    Halt(1);
  end;
  
  WriteLn('Connected to X server');
  CreateWindow;
  WriteLn('Window created. Click on it or press keys.');
  WriteLn('Will show 20 events then exit.');
  WriteLn;
  
  // Самый безопасный подход: не освобождаем события вообще
  while event_count < 20 do
  begin
    event := xcb_poll_for_event(conn);
    
    if event <> nil then
    begin
      Inc(event_count);
      WriteLn('Event ', event_count:2, ': ', 
              GetEventName(event^.response_type and $7F),
              ' (Sequence: ', event^.sequence, ')');
      
      // НИКОГДА не освобождаем события в этом демо
      // Это безопасно для коротких программ
    end
    else
    begin
      Sleep(50);
    end;
  end;
  
  WriteLn;
  WriteLn('Reached event limit. Exiting safely.');
  xcb_destroy_window(conn, window);
  xcb_disconnect(conn);
  WriteLn('Disconnected from X server');
  WriteLn('Demo completed without crashes!');
end.