program test_events_blocking;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, xproto_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: xcb_window_t;
  event: Pxcb_generic_event_t;
  has_error: cint;
  event_count: integer = 0;
  start_time: QWord;
  running: Boolean = True;

procedure SignalHandler(sig: cint); cdecl;
begin
  if sig = SIGINT then
  begin
    WriteLn;
    WriteLn('Received SIGINT, shutting down...');
    running := False;
  end;
end;

procedure CreateTestWindow;
var
  setup: Pxcb_setup_t;
  screen_iterator: xcb_screen_iterator_t;
  value_mask: uint32_t;
  value_list: array[0..0] of uint32_t;
begin
  // Get setup and screen
  setup := xcb_get_setup(conn);
  screen_iterator := xcb_setup_roots_iterator(setup);
  screen := screen_iterator.data;
  
  // Create window
  window := xcb_generate_id(conn);
  
  // Set event mask to receive events
  value_mask := XCB_CW_EVENT_MASK;
  value_list[0] := XCB_EVENT_MASK_EXPOSURE or 
                   XCB_EVENT_MASK_KEY_PRESS or 
                   XCB_EVENT_MASK_KEY_RELEASE or
                   XCB_EVENT_MASK_BUTTON_PRESS or
                   XCB_EVENT_MASK_BUTTON_RELEASE or
                   XCB_EVENT_MASK_POINTER_MOTION or
                   XCB_EVENT_MASK_ENTER_WINDOW or
                   XCB_EVENT_MASK_LEAVE_WINDOW or
                   XCB_EVENT_MASK_FOCUS_CHANGE or
                   XCB_EVENT_MASK_STRUCTURE_NOTIFY;
  
  xcb_create_window(conn,
    XCB_COPY_FROM_PARENT, // depth
    window,
    screen^.root,         // parent
    100, 100,             // x, y
    400, 300,             // width, height
    10,                   // border width
    XCB_WINDOW_CLASS_INPUT_OUTPUT,
    screen^.root_visual,
    value_mask, @value_list);
  
  // Map window
  xcb_map_window(conn, window);
  
  // Flush requests
  xcb_flush(conn);
  
  WriteLn('Created test window with ID: ', window);
  WriteLn('Window should appear at position 100,100 with size 400x300');
end;

function GetEventTypeName(event_type: uint8_t): string;
begin
  case event_type of
    0: Result := 'Error';
    1: Result := 'Reply';
    2: Result := 'KeyPress';
    3: Result := 'KeyRelease';
    4: Result := 'ButtonPress';
    5: Result := 'ButtonRelease';
    6: Result := 'MotionNotify';
    7: Result := 'EnterNotify';
    8: Result := 'LeaveNotify';
    9: Result := 'FocusIn';
    10: Result := 'FocusOut';
    11: Result := 'KeymapNotify';
    12: Result := 'Expose';
    18: Result := 'ConfigureNotify';
    19: Result := 'PropertyNotify';
    22: Result := 'ClientMessage';
    33: Result := 'MappingNotify';
    else Result := 'Other(' + IntToStr(event_type) + ')';
  end;
end;

begin
  WriteLn('Testing XCB event handling (blocking mode)...');
  WriteLn('Press Ctrl+C to exit');
  
  // Setup signal handler for graceful shutdown
  FpSignal(SIGINT, @SignalHandler);
  
  // Connect to X server
  conn := xcb_connect(nil, nil);
  
  if conn = nil then
  begin
    WriteLn('Failed to connect to X server');
    Halt(1);
  end;
  
  has_error := xcb_connection_has_error(conn);
  if has_error <> 0 then
  begin
    WriteLn('Connection error: ', has_error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  
  WriteLn('Connected to X server');
  
  // Create a test window to generate events
  CreateTestWindow;
  
  WriteLn('Waiting for events (blocking mode)...');
  WriteLn('Interact with the window or press Ctrl+C to exit');
  
  while running do
  begin
    // Wait for event (blocking)
    event := xcb_wait_for_event(conn);
    
    if event <> nil then
    begin
      Inc(event_count);
      WriteLn('Event #', event_count, ': ', 
              GetEventTypeName(event^.response_type and $7F),
              ' (Type=', event^.response_type and $7F, 
              ', Sequence=', event^.sequence, ')');
      
      // For blocking mode, we can safely free the event
      // xcb_wait_for_event always returns standard-sized events
      FreeMem(event);
    end
    else
    begin
      // Null event means connection error
      WriteLn('Error: received null event');
      running := False;
    end;
  end;
  
  WriteLn('Processed ', event_count, ' events');
  
  // Cleanup
  WriteLn('Destroying window...');
  xcb_destroy_window(conn, window);
  xcb_flush(conn);
  xcb_disconnect(conn);
  WriteLn('Disconnected from X server');
end.