program test_events_simple;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  event: Pxcb_generic_event_t;
  has_error: cint;
  event_count: integer = 0;
  running: Boolean = True;

procedure SignalHandler(sig: cint); cdecl;
begin
  if sig = SIGINT then
  begin
    WriteLn;
    WriteLn('Received SIGINT, shutting down...');
    running := False;
  end;
end;

function GetEventTypeName(event_type: uint8_t): string;
begin
  case event_type of
    0: Result := 'Error';
    1: Result := 'Reply';
    2: Result := 'KeyPress';
    3: Result := 'KeyRelease';
    4: Result := 'ButtonPress';
    5: Result := 'ButtonRelease';
    6: Result := 'MotionNotify';
    7: Result := 'EnterNotify';
    8: Result := 'LeaveNotify';
    9: Result := 'FocusIn';
    10: Result := 'FocusOut';
    11: Result := 'KeymapNotify';
    12: Result := 'Expose';
    18: Result := 'ConfigureNotify';
    19: Result := 'PropertyNotify';
    22: Result := 'ClientMessage';
    33: Result := 'MappingNotify';
    else Result := 'Other(' + IntToStr(event_type) + ')';
  end;
end;

begin
  WriteLn('Simple XCB event test');
  WriteLn('Press Ctrl+C to exit');
  WriteLn('This will show system-wide X events');
  
  // Setup signal handler
  FpSignal(SIGINT, @SignalHandler);
  
  // Connect to X server
  conn := xcb_connect(nil, nil);
  
  if conn = nil then
  begin
    WriteLn('Failed to connect to X server');
    Halt(1);
  end;
  
  has_error := xcb_connection_has_error(conn);
  if has_error <> 0 then
  begin
    WriteLn('Connection error: ', has_error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  
  WriteLn('Connected to X server - listening for events...');
  
  while running do
  begin
    // Use non-blocking poll
    event := xcb_poll_for_event(conn);
    
    if event <> nil then
    begin
      Inc(event_count);
      Write('Event #', event_count, ': ', 
            GetEventTypeName(event^.response_type and $7F));
      WriteLn(' (Seq=', event^.sequence, ')');
      
      // Правильное освобождение события
      xcb_free_event(event);
    end
    else
    begin
      // No events, small delay to avoid busy waiting
      Sleep(50);
    end;
  end;
  
  WriteLn('Processed ', event_count, ' events');
  xcb_disconnect(conn);
  WriteLn('Disconnected from X server');
end.