program test_safe_events;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb_bindings, BaseUnix, SysUtils;

var
  conn: Pxcb_connection_t;
  event: Pxcb_generic_event_t;
  has_error: cint;
  event_count: integer = 0;

function GetEventTypeName(event_type: uint8_t): string;
begin
  case event_type of
    0: Result := 'Error';
    1: Result := 'Reply';
    2: Result := 'KeyPress';
    3: Result := 'KeyRelease';
    4: Result := 'ButtonPress';
    5: Result := 'ButtonRelease';
    6: Result := 'MotionNotify';
    7: Result := 'EnterNotify';
    8: Result := 'LeaveNotify';
    9: Result := 'FocusIn';
    10: Result := 'FocusOut';
    11: Result := 'KeymapNotify';
    12: Result := 'Expose';
    18: Result := 'ConfigureNotify';
    19: Result := 'PropertyNotify';
    22: Result := 'ClientMessage';
    33: Result := 'MappingNotify';
    else Result := 'Other(' + IntToStr(event_type) + ')';
  end;
end;

begin
  WriteLn('Safe XCB event test');
  WriteLn('Testing xcb_poll_for_queued_event (safe method)');
  WriteLn('This only shows already queued events, no new events will be generated');
  
  // Connect to X server
  conn := xcb_connect(nil, nil);
  
  if conn = nil then
  begin
    WriteLn('Failed to connect to X server');
    Halt(1);
  end;
  
  has_error := xcb_connection_has_error(conn);
  if has_error <> 0 then
  begin
    WriteLn('Connection error: ', has_error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  
  WriteLn('Connected to X server');
  WriteLn('Checking for already queued events...');
  
  // xcb_poll_for_queued_event only returns events already in the queue
  // It doesn't read from the connection, so it's safer
  event := xcb_poll_for_queued_event(conn);
  
  while event <> nil do
  begin
    Inc(event_count);
    WriteLn('Queued Event #', event_count, ': ', 
            GetEventTypeName(event^.response_type and $7F),
            ' (Type=', event^.response_type and $7F, 
            ', Sequence=', event^.sequence, ')');
    
    // Освобождаем событие
    FreeMem(event);
    
    // Получаем следующее событие из очереди
    event := xcb_poll_for_queued_event(conn);
  end;
  
  if event_count = 0 then
    WriteLn('No queued events found');
  
  WriteLn('Processed ', event_count, ' queued events');
  xcb_disconnect(conn);
  WriteLn('Disconnected from X server');
  WriteLn('Safe test completed!');
end.