program xcb_clock_example;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}
{$PACKRECORDS C}
{$GOTO ON}

uses
  SysUtils, ctypes, BaseUnix,
  xcb_bindings, xproto_bindings, xcb_more_bindings,xcb_events_bindings,xcb_const;

{ --- Дополнительные типы (если дублируются с вашими модулями — удалите локальные объявления) --- }

type
  xcb_font_t = uint32_t;
  Pxcb_rectangle_t = ^xcb_rectangle_t;
  xcb_rectangle_t = record
    x: int16_t;
    y: int16_t;
    width: uint16_t;
    height: uint16_t;
  end;

{ --- Константы GC (если они уже есть в биндингах — можно удалить) --- }
{
const
  libxcb = 'libxcb.so.1';

  XCB_GC_FOREGROUND = 1 shl 2;
  XCB_GC_BACKGROUND = 1 shl 3;
  XCB_GC_LINE_WIDTH = 16;
  XCB_GC_LINE_STYLE = 32;
  XCB_GC_FONT       = 16384;
}

const
  WM_PROTOCOLS_STR = 'WM_PROTOCOLS';
  WM_DELETE_WINDOW_STR = 'WM_DELETE_WINDOW';

{ --- Внешние функции (если они уже объявлены в биндингах — удалить локальные объявления) --- }
function xcb_open_font(conn: Pxcb_connection_t; fid: xcb_font_t; name_len: uint16_t; name: PChar): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_open_font';
function xcb_close_font(conn: Pxcb_connection_t; font: xcb_font_t): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_close_font';

function xcb_create_gc(conn: Pxcb_connection_t; cid: xcb_gcontext_t; drawable: cuint32;
  value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_create_gc';

function xcb_change_gc(conn: Pxcb_connection_t; cid: xcb_gcontext_t;
  value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_change_gc';

function xcb_poly_fill_rectangle(conn: Pxcb_connection_t; drawable: cuint32; gc: xcb_gcontext_t;
  rectangles_len: cuint; rectangles: Pxcb_rectangle_t): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_poly_fill_rectangle';

function xcb_image_text_8(conn: Pxcb_connection_t; string_len: cuint; drawable: cuint32; gc: xcb_gcontext_t; x: int16_t; y: int16_t; text: PChar): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_image_text_8';

{ POSIX threads / sleep }
type
  pthread_t = QWord;
  Ppthread_t = ^pthread_t;

function pthread_create(thread: Ppthread_t; attr: Pointer; start_routine: Pointer; arg: Pointer): cint; cdecl; external 'pthread' name 'pthread_create';
function sleep(seconds: cuint): cuint; cdecl; external 'c' name 'sleep';

{ --- Утилита RGB -> pixel (24-bit) --- }
function RGBToPixel(r, g, b: cint): uint32_t;
begin
  Result := ((uint32_t(r) and $FF) shl 16) or ((uint32_t(g) and $FF) shl 8) or (uint32_t(b) and $FF);
end;

label _exitloop;

{ --- Глобальные состояния --- }
var
  conn: Pxcb_connection_t = nil;
  screen: Pxcb_screen_t = nil;
  win: xcb_window_t = 0;
  gc: xcb_gcontext_t = 0;
  font: xcb_font_t = 0;
  pipe_fds: array[0..1] of cint;
  wm_delete_atom: xcb_atom_t = 0;
  wm_protocols, wm_delete_window: xcb_atom_t;
  reply: Pxcb_intern_atom_reply_t;
  msg: Pxcb_client_message_event_t;

{ --- Вспомогательная процедура для установки FG в GC (value как uint32_t) --- }
procedure SetGCForeground(conn: Pxcb_connection_t; gc: xcb_gcontext_t; color: uint32_t);
var
  val: array[0..0] of uint32_t;
begin
  val[0] := color;
  xcb_change_gc(conn, gc, XCB_GC_FOREGROUND, @val[0]);
end;

{ --- Процедура рисования часов и календаря --- }
procedure DrawAll();
var
  r: TDateTime;
  sTime, sDate: string;
  rect: xcb_rectangle_t;
  black, white, red, blue, gray: uint32_t;
  yText, xText: cint;
begin
  if (conn = nil) or (screen = nil) then Exit;

  r := Now;
  sTime := FormatDateTime('HH:nn:ss', r);
  sDate := FormatDateTime('dddd, DD MMMM YYYY', r);

  black := RGBToPixel(0,0,0);
  white := RGBToPixel(255,255,255);
  red   := RGBToPixel(200,20,20);
  blue  := RGBToPixel(30,90,200);
  gray  := RGBToPixel(230,230,230);

  // fill background white
  rect.x := 0; rect.y := 0;
  rect.width := screen^.width_in_pixels;
  rect.height := screen^.height_in_pixels;
  SetGCForeground(conn, gc, white);
  xcb_poly_fill_rectangle(conn, win, gc, 1, @rect);

  // time in blue
  SetGCForeground(conn, gc, blue);
  xText := 40;
  yText := 120;
  xcb_image_text_8(conn, Length(sTime), win, gc, xText, yText, PChar(sTime));

  // date in red
  SetGCForeground(conn, gc, red);
  xcb_image_text_8(conn, Length(sDate), win, gc, xText, yText + 40, PChar(sDate));

  // small calendar box (gray background)
  rect.x := screen^.width_in_pixels - 260 - 1600;
  rect.y := 20 + 220;
  rect.width := 240;
  rect.height := 180;
  SetGCForeground(conn, gc, gray);
  xcb_poly_fill_rectangle(conn, win, gc, 1, @rect);

  // calendar text inside
  SetGCForeground(conn, gc, black);
  xcb_image_text_8(conn, Length('Calendar:'), win, gc, rect.x + 10, rect.y + 30, PChar('Calendar:'));
  xcb_image_text_8(conn, Length('Mon Tue Wed Thu Fri Sat Sun'), win, gc, rect.x + 10, rect.y + 60, PChar('Mon Tue Wed Thu Fri Sat Sun'));

  xcb_flush(conn);
end;

{ --- Поток 'тиканья' (POSIX thread) --- }
function TickerThreadProc(arg: Pointer): Pointer; cdecl;
var
  b: array[0..0] of Byte;
begin
  // бесконечный цикл: пишем 1 байт в pipe каждую секунду
  while True do
  begin
    b[0] := 1;
    fpWrite(pipe_fds[1], @b[0], 1);
    sleep(1);
  end;
  Result := nil;
end;

{ --- Main --- }
var
  scr_iter: xcb_screen_iterator_t;
  setup: Pxcb_setup_t;
  fdset: TFDSet;
  maxfd, s: cint;
  buf: array[0..0] of Byte;
  ev: Pxcb_generic_event_t;
  cookie: xcb_void_cookie_t;
  screen_num: cint = 0;
  thr: pthread_t;
  event_mask: uint32_t;
  value_list: array[0..1] of uint32_t;
  attr_vals: array[0..0] of uint32_t;
  gc_vals: array[0..1] of uint32_t;
  font_val: array[0..0] of uint32_t;
  pthread_ret: cint;
begin
  // create pipe
  if fpPipe(pipe_fds) <> 0 then
  begin
    writeln('pipe() failed: ', fpgeterrno);
    Halt(1);
  end;

  // connect to X
  conn := xcb_connect(nil, @screen_num);
  if conn = nil then
  begin
    writeln('xcb_connect returned nil');
    Halt(1);
  end;
  if xcb_connection_has_error(conn) <> 0 then
  begin
    writeln('Cannot connect to X (xcb_connection_has_error)');
    Halt(1);
  end;

  // setup & screen
  setup := xcb_get_setup(conn);
  scr_iter := xcb_setup_roots_iterator(setup);
  screen := scr_iter.data;
  if screen = nil then
  begin
    writeln('No screen found');
    Halt(1);
  end;


  // create window
  win := xcb_generate_id(conn);

  event_mask := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_KEY_PRESS or XCB_EVENT_MASK_BUTTON_PRESS or XCB_EVENT_MASK_STRUCTURE_NOTIFY;
  value_list[0] := screen^.white_pixel;
  value_list[1] := event_mask;

  cookie := xcb_create_window(conn, screen^.root_depth, win, screen^.root,
    100, 100, 640, 360, 0,
    XCB_WINDOW_CLASS_INPUT_OUTPUT, screen^.root_visual,
    XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK, @value_list[0]);


  reply := xcb_intern_atom_reply(conn, xcb_intern_atom(conn, 1, Length(WM_PROTOCOLS_STR), PChar(WM_PROTOCOLS_STR)), nil);
  wm_protocols := reply^.atom;
  CFree(reply);
  reply := xcb_intern_atom_reply(conn, xcb_intern_atom(conn, 0, Length(WM_DELETE_WINDOW_STR), PChar(WM_DELETE_WINDOW_STR)), nil);
  wm_delete_window := reply^.atom;
  CFree(reply);


   // WM_DELETE_WINDOW registration
 //  wm_delete_atom := GetAtom(conn, 'WM_DELETE_WINDOW');
 //  SetWMProtocols_DeleteWindow(conn, win);

//  wm_protocols := GetAtom(conn, 'WM_PROTOCOLS');
//  wm_delete_window := GetAtom(conn, 'WM_DELETE_WINDOW');

  // Говорим окну, что поддерживаем WM_DELETE_WINDOW
  xcb_change_property(conn, XCB_PROP_MODE_REPLACE, win, wm_protocols, XCB_ATOM_ATOM, 32, 1, @wm_delete_window);



  // set event mask explicitly (some servers need separate change)
  attr_vals[0] := event_mask;
  xcb_change_window_attributes(conn, win, XCB_CW_EVENT_MASK, @attr_vals[0]);

  xcb_map_window(conn, win);
  xcb_flush(conn);

  // create GC
  gc := xcb_generate_id(conn);
  gc_vals[0] := screen^.black_pixel; // foreground
  gc_vals[1] := screen^.white_pixel; // background
  xcb_create_gc(conn, gc, win, XCB_GC_FOREGROUND or XCB_GC_BACKGROUND, @gc_vals[0]);

  // try to open font "fixed" (optional)
  font := xcb_generate_id(conn);
//  xcb_open_font(conn, font, Length('fixed'), PChar('fixed'));
//  xcb_open_font(conn, font, Length('*helvetica*'), PChar('*helvetica*'));
  xcb_open_font(conn, font, Length('-*-fixed-bold-r-normal--50-*-*-*-*-*-iso10646-1'), PChar('-*-fixed-bold-r-normal--50-*-*-*-*-*-iso10646-1'));
//  xcb_open_font(conn, font, Length('*terminal*'), PChar('*terminal*'));
  // set font into GC via change_gc (if the server and GC support it)
  // We won't error out if it fails — image_text_8 will use default font
  font_val[0] := font;
  // change font in GC (mask XCB_GC_FONT)
  xcb_change_gc(conn, gc, XCB_GC_FONT, @font_val[0]);
  xcb_flush(conn);

  // start ticker thread
  thr := 0;
  pthread_ret := pthread_create(@thr, nil, @TickerThreadProc, nil);
  if pthread_ret <> 0 then
    writeln('pthread_create failed: ', pthread_ret);

  // prepare select maxfd
  maxfd := xcb_get_file_descriptor(conn);
  if pipe_fds[0] > maxfd then maxfd := pipe_fds[0];
  Inc(maxfd); // select expects maxfd+1

  // initial paint
  DrawAll();

  // main loop: select on xcb fd and pipe
  while True do
  begin
    fpFD_ZERO(fdset);
    fpFD_SET(xcb_get_file_descriptor(conn), fdset);
    fpFD_SET(pipe_fds[0], fdset);

    s := fpSelect(maxfd, @fdset, nil, nil, nil);
    if s < 0 then
    begin
      writeln('select error: ', fpgeterrno);
      Break;
    end;

    // X events
    if boolean( fpFD_ISSET(xcb_get_file_descriptor(conn), fdset) ) then
    begin
      // process all pending events
      while True do
      begin
        ev := xcb_poll_for_event(conn);
        if ev = nil then Break;

        case ev^.response_type of
          XCB_EXPOSE:
            begin
              DrawAll();
            end;
          XCB_KEY_PRESS:
            begin
              // If keycode == 24 (common 'q') -> exit (heuristic)
              if Pxcb_key_press_event_t(ev)^.detail = 24 then
              begin
                cfree(ev);
                goto _exitloop;
              end;
            end;
//          33: // ClientMessage (usually WM protocols)
//            begin
//              // Could inspect client message for WM_DELETE_WINDOW; for simplicity, we exit when client asked to close
//              // Full implementation should check message type and atom.
//              goto _exitloop;
//            end;
//XCB_CLIENT_MESSAGE: begin
//writeln('XCB_CLIENT_MESSAGE');
//  msg := Pxcb_client_message_event_t(ev);
//  if (msg^._type = wm_protocols) and
//     (msg^.data.data32[0] = wm_delete_window) then
//  begin
//    writeln('WM_DELETE_WINDOW → exiting');
//    break;
//  end;
//end;
161: begin
  msg := Pxcb_client_message_event_t(ev);
  if (msg^._type = wm_protocols) and
     (msg^.data.data32[0] = wm_delete_window) then
  begin
    writeln('WM_DELETE_WINDOW → exiting');
//    break;
    goto _exitloop;
  end;
end;
//else
//WriteLn('ev^.response_type = ', ev^.response_type);
        end;

        cfree(ev);
      end;
    end;

    // pipe (tick) readable
    if boolean( fpFD_ISSET(pipe_fds[0], fdset) ) then
    begin
      if fpRead(pipe_fds[0], @buf[0], 1) > 0 then
      begin
        DrawAll();
      end;
    end;
  end;

_exitloop:
  // cleanup
  if font <> 0 then xcb_close_font(conn, font);
  // no explicit free for gc in this example
  xcb_disconnect(conn);
  writeln('Exited');
end.
