program xcb_clock_xrender;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}
{$PACKRECORDS C}
{$INLINE ON}
{$OPTIMIZATION LEVEL3}
{$GOTO ON}

uses
  pthreads,SysUtils, ctypes, BaseUnix, xcb_const, xcb_bindings, xcb_events_bindings, xcb_ext_bindings, xcb_more_bindings, xproto_bindings,
  ucs4unit, ucs4opunit,Math; // ваши юниты для UTF-8 -> UCS4

{ --------------------------------------------------------------------
  ПРИМЕЧАНИЕ:
  Если в вашей кодовой базе уже есть объявления некоторых XCB/KeySym/FreeType
  типов или функций — удалите дублирующие блоки ниже (я пометил их комментариями).
--------------------------------------------------------------------- }

const
  libxcb = 'libxcb.so.1';
  libfreetype = 'libfreetype.so.6';
  libxcb_keysyms = 'libxcb-keysyms.so.1';
  // путь к TTF-шрифту
  FontPath: PChar = '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf';

  FT_LOAD_RENDER = 4; // FreeType flag to render glyph bitmap during load

type
  // -- XCB базовые типы (если у вас уже есть в xcb_bindings, уберите дубликаты) --
  uint8_t = cuint8;
  uint16_t = cuint16;
  uint32_t = cuint32;
  int32_t = cint32;
  uint64_t = cuint64;

  int16_t = cint16;

  xcb_connection_t = record end;
  Pxcb_connection_t = ^xcb_connection_t;

  xcb_window_t = uint32_t;
  xcb_gcontext_t = uint32_t;
  xcb_atom_t = uint32_t;
  xcb_visualid_t = uint32_t;

  Pxcb_generic_event_t = ^xcb_generic_event_t;
  xcb_generic_event_t = record
    response_type: uint8_t;
    pad0: uint8_t;
    sequence: uint16_t;
    pad: array[0..6] of uint32_t;
    full_sequence: uint32_t;
  end;

  Pxcb_key_press_event_t = ^xcb_key_press_event_t;
  xcb_key_press_event_t = record
    response_type: uint8_t;
    detail: uint8_t;
    sequence: uint16_t;
    time: uint32_t;
    root: xcb_window_t;
    event: xcb_window_t;
    child: xcb_window_t;
    root_x: int16_t;
    root_y: int16_t;
    event_x: int16_t;
    event_y: int16_t;
    state: uint16_t;
    same_screen: uint8_t;
    pad0: uint8_t;
  end;

  // ClientMessage types
  xcb_client_message_data_t = record
    case Integer of
      0: (data8: array[0..19] of cuint8);
      1: (data16: array[0..9] of cuint16);
      2: (data32: array[0..4] of cuint32);
  end;

  xcb_client_message_event_t = record
    response_type: cuint8;
    format: cuint8;
    sequence: cuint16;
    window: xcb_window_t;
    event_type: xcb_atom_t; // called "type" in C
    data: xcb_client_message_data_t;
  end;
  Pxcb_client_message_event_t = ^xcb_client_message_event_t;

  // get setup / screen (reduced)
  Pxcb_setup_t = ^xcb_setup_t;
  xcb_setup_t = record
    pad: array[0..0] of uint8_t; // not used fully here
  end;

  Pxcb_screen_t = ^xcb_screen_t;
  xcb_screen_t = record
    root: xcb_window_t;
    default_colormap: uint32_t;
    white_pixel: uint32_t;
    black_pixel: uint32_t;
    current_input_masks: uint32_t;
    width_in_pixels: uint16_t;
    height_in_pixels: uint16_t;
    width_in_millimeters: uint16_t;
    height_in_millimeters: uint16_t;
    min_installed_maps: uint16_t;
    max_installed_maps: uint16_t;
    root_visual: xcb_visualid_t;
    backing_stores: uint8_t;
    save_unders: uint8_t;
    root_depth: uint8_t;
    allowed_depths_len: uint8_t;
  end;

  xcb_screen_iterator_t = record
    data: Pxcb_screen_t;
    rem: cint;
    index: cint;
  end;

  // cookies / replies simplified
  xcb_void_cookie_t = record sequence: cuint; end;

{ --------------------------------------------------------------------
  Внешние функции libxcb, libxcb-keysyms и libfreetype (минимально необходимые)
--------------------------------------------------------------------- }
function xcb_connect(displayname: PChar; screenp: Pcint): Pxcb_connection_t; cdecl; external libxcb;
procedure xcb_disconnect(c: Pxcb_connection_t); cdecl; external libxcb;
function xcb_connection_has_error(c: Pxcb_connection_t): cint; cdecl; external libxcb;
function xcb_get_setup(c: Pxcb_connection_t): Pxcb_setup_t; cdecl; external libxcb;
function xcb_setup_roots_iterator(setup: Pxcb_setup_t): xcb_screen_iterator_t; cdecl; external libxcb;
function xcb_generate_id(c: Pxcb_connection_t): uint32_t; cdecl; external libxcb;
function xcb_get_file_descriptor(c: Pxcb_connection_t): cint; cdecl; external libxcb;

function xcb_create_window(connection: Pxcb_connection_t; depth: uint8_t; window, parent: xcb_window_t;
  x, y, width, height: int16_t; border_width, _class: uint16_t; visual: xcb_visualid_t;
  value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_create_window';

function xcb_map_window(connection: Pxcb_connection_t; window: xcb_window_t): xcb_void_cookie_t; cdecl; external libxcb;
function xcb_change_window_attributes(conn: Pxcb_connection_t; window: xcb_window_t; value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb;

function xcb_create_gc(conn: Pxcb_connection_t; cid: xcb_gcontext_t; drawable: cuint32;
  value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb;

function xcb_put_image(conn: Pxcb_connection_t; format: uint8_t; drawable: cuint32; gc: xcb_gcontext_t;
  width: uint16_t; height: uint16_t; dst_x: int16_t; dst_y: int16_t; left_pad: uint8_t;
  depth: uint8_t; size: cuint; data: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_put_image';

function xcb_change_property(conn: Pxcb_connection_t; mode: uint8_t; window: xcb_window_t;
  _property: xcb_atom_t; type_: xcb_atom_t; format: uint8_t; data_len: cuint; data: Pointer): xcb_void_cookie_t; cdecl; external libxcb;

function xcb_intern_atom(conn: Pxcb_connection_t; only_if_exists: uint8_t; name_len: uint16_t; name: PChar): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_intern_atom';
function xcb_intern_atom_reply(conn: Pxcb_connection_t; cookie: xcb_void_cookie_t; e: PPxcb_generic_event_t): Pointer; cdecl; external libxcb name 'xcb_intern_atom_reply';

function xcb_poll_for_event(c: Pxcb_connection_t): Pxcb_generic_event_t; cdecl; external libxcb name 'xcb_poll_for_event';
procedure xcb_flush(c: Pxcb_connection_t); cdecl; external libxcb;

function xcb_open_font(conn: Pxcb_connection_t; fid: uint32_t; name_len: uint16_t; name: PChar): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_open_font';
function xcb_close_font(conn: Pxcb_connection_t; font: uint32_t): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_close_font';

{ xcb-keysyms }
type
  Pxcb_key_symbols_t = Pointer;
  xcb_keysym_t = cuint32;

function xcb_key_symbols_alloc(conn: Pxcb_connection_t): Pxcb_key_symbols_t; cdecl; external libxcb_keysyms name 'xcb_key_symbols_alloc';
function xcb_key_symbols_get_keysym(keysyms: Pxcb_key_symbols_t; keycode: cuint8; index: cint): xcb_keysym_t; cdecl; external libxcb_keysyms name 'xcb_key_symbols_get_keysym';
procedure xcb_key_symbols_free(keysyms: Pxcb_key_symbols_t); cdecl; external libxcb_keysyms name 'xcb_key_symbols_free';

{ malloc/free }
procedure cfree(p: Pointer); cdecl; external 'c' name 'free';

{ FreeType minimal bindings (наиболее используемые функции и структуры) }
type
  FT_Library = Pointer;
  PFT_Library = ^FT_Library;
  FT_Face = Pointer;

  TFT_Bitmap = record
    rows: cint;
    width: cint;
    pitch: cint;
    buffer: Pointer;
    num_grays: cint;
    pixel_mode: cint;
    palette: Pointer;
  end;
  PFT_Bitmap = ^TFT_Bitmap;

  TFT_Glyph_Metrics = record
    width: LongInt;
    height: LongInt;
    horiBearingX: LongInt;
    horiBearingY: LongInt;
    horiAdvance: LongInt;
    vertBearingX: LongInt;
    vertBearingY: LongInt;
    vertAdvance: LongInt;
  end;

  TFT_GlyphSlot = record
    advance: record x, y: LongInt; end;
    metrics: TFT_Glyph_Metrics;
    bitmap: TFT_Bitmap;
    bitmap_left: LongInt;
    bitmap_top: LongInt;
  end;

{
  PFT_GlyphSlot = ^TFT_GlyphSlot;

  TFT_FaceRec = record
    num_faces: LongInt;
    face_index: LongInt;
    face_flags: LongInt;
    style_flags: LongInt;
    num_glyphs: LongInt;
    family_name: PChar;
    style_name: PChar;
    // ... we only need glyph slot pointer
    // typical layout: after many fields there is glyph slot pointer; we'll obtain via FT_Get_GlyphSlot via face^? but simpler: we call FT_Load_Char and then FT_Get_Glyph_SLOT function
  end;
  PFT_FaceRec = ^TFT_FaceRec;
}
  PFT_GlyphSlot = ^TFT_GlyphSlot;
  TFT_FaceRec = record
    num_faces: LongInt;
    face_index: LongInt;
    face_flags: LongInt;
    style_flags: LongInt;
    num_glyphs: LongInt;
    family_name: PChar;
    style_name: PChar;
    // ... (возможно другие поля, не обязательно перечислять все)
    // Важное поле:
    glyph: PFT_GlyphSlot; // -> this points to the current glyph slot after FT_Load_* calls
  end;
  PFT_FaceRec = ^TFT_FaceRec;

  PFT_Face = FT_Face;

function FT_Init_FreeType(alibrary: PFT_Library): cint; cdecl; external libfreetype name 'FT_Init_FreeType';
function FT_New_Face(_library: FT_Library; filepathname: PChar; face_index: cint; aface: PFT_Face): cint; cdecl; external libfreetype name 'FT_New_Face';
function FT_Done_Face(face: FT_Face): cint; cdecl; external libfreetype name 'FT_Done_Face';
function FT_Done_FreeType(_library: FT_Library): cint; cdecl; external libfreetype name 'FT_Done_FreeType';
function FT_Set_Pixel_Sizes(face: FT_Face; pixel_width: cuint; pixel_height: cuint): cint; cdecl; external libfreetype name 'FT_Set_Pixel_Sizes';
function FT_Load_Char(face: FT_Face; charcode: LongInt; load_flags: cint): cint; cdecl; external libfreetype name 'FT_Load_Char';
//function FT_Get_GlyphSlot(face: FT_Face): PFT_GlyphSlot; cdecl; external libfreetype name 'FT_Get_Glyph_Slot'; // some builds export FT_Get_Glyph_Slot
function FT_Get_GlyphSlot(face: FT_Face): PFT_GlyphSlot; cdecl; external libfreetype name 'FT_Get_Glyph'; // some builds export FT_Get_Glyph_Slot

{ Fallback: if FT_Get_GlyphSlot isn't present we will access face pointer as unknown; hope FT_Get_Glyph_Slot exists. }

{ Constants }
const
  XCB_EVENT_MASK_EXPOSURE = 1 shl 15;
  XCB_EVENT_MASK_KEY_PRESS = 1 shl 0;
  XCB_EVENT_MASK_BUTTON_PRESS = 1 shl 2;
  XCB_EXPOSE = 12;
  XCB_KEY_PRESS = 2;
  XCB_BUTTON_PRESS = 4;
  XCB_CLIENT_MESSAGE = 33;

  XCB_CW_BACK_PIXEL = 1 shl 1;
  XCB_CW_EVENT_MASK = 1 shl 11;

  XCB_WINDOW_CLASS_INPUT_OUTPUT = 1;

  XCB_PROP_MODE_REPLACE = 0;

{ Simple glyph cache entry }
type
  TGlyphCacheEntry = record
    codepoint: DWord;
    width: Integer;
    height: Integer;
    left, top: Integer;
    advance: Integer;
    // ARGB buffer
    pixels: Pointer; // points to width*height*4 bytes (ARGB32)
  end;
  PGlyphCacheEntry = ^TGlyphCacheEntry;

{ Globals }
var
  conn: Pxcb_connection_t = nil;
  screen: Pxcb_screen_t = nil;
  win: xcb_window_t = 0;
  gc: xcb_gcontext_t = 0;
  pipe_fds: array[0..1] of cint;
  fontLib: FT_Library = nil;
  fontFace: FT_Face = nil;
  FontPixelSize: Integer = 96; // big font
  keysyms: Pxcb_key_symbols_t = nil;
  wm_protocols_atom: xcb_atom_t = 0;
  wm_delete_atom: xcb_atom_t = 0;

  // very small glyph cache (linear probe)
  GlyphCache: array of TGlyphCacheEntry;

{ Helpers: memory management for ARGB buffers }
function AllocARGB(w, h: Integer): Pointer;
var
  size: NativeUint;
begin
  size := NativeUint(w) * NativeUint(h) * 4;
  GetMem(Result, size);
  FillChar(Result^, size, 0); // transparent
end;

procedure FreeARGB(p: Pointer);
begin
  if p <> nil then FreeMem(p);
end;

{ Find glyph in cache (returns index or -1) }
function GlyphCacheFind(cp: DWord): Integer;
var
  i: Integer;
begin
  for i := 0 to High(GlyphCache) do
    if GlyphCache[i].codepoint = cp then Exit(i);
  Result := -1;
end;

{ Add glyph to cache (returns index) }
function GlyphCacheAdd(entry: TGlyphCacheEntry): Integer;
var
  i: Integer;
  e: TGlyphCacheEntry;
begin
  // find empty slot
  for i := 0 to High(GlyphCache) do
  begin
    if GlyphCache[i].codepoint = 0 then
    begin
      GlyphCache[i] := entry;
      Exit(i);
    end;
  end;
  // grow
  i := Length(GlyphCache);
  SetLength(GlyphCache, Max(8, i*2));
  // init new slots' codepoint=0
  if i = 0 then i := 0;
  GlyphCache[High(GlyphCache)] := entry;
  Result := High(GlyphCache);
end;

{ Free entire glyph cache }
procedure GlyphCacheClear();
var i: Integer;
begin
  for i := 0 to High(GlyphCache) do
    if GlyphCache[i].codepoint <> 0 then
    begin
      FreeARGB(GlyphCache[i].pixels);
      GlyphCache[i].codepoint := 0;
    end;
  SetLength(GlyphCache, 0);
end;

{ Load and rasterize glyph for codepoint cp, color applied later.
  Returns cache index (>=0) or -1 on error. }
function RasterizeGlyph(cp: DWord; out pixW, pixH, left, top, advance: Integer): Integer;
var
  err: cint;
  slot: PFT_GlyphSlot = nil;
  bmp: TFT_Bitmap;
  i, x, y: Integer;
  dst: PByte;
  argb: PByte;
  cacheEntry: TGlyphCacheEntry;
  rowBytes: NativeUint;
  bmpRow: PByte;
cov: Byte = 0;
faceRec: PFT_FaceRec;
begin
{
  Result := GlyphCacheFind(cp);
  if Result >= 0 then
  begin
    pixW := GlyphCache[Result].width;
    pixH := GlyphCache[Result].height;
    left := GlyphCache[Result].left;
    top := GlyphCache[Result].top;
    advance := GlyphCache[Result].advance;
    Exit;
  end;

  // load glyph and render
  err := FT_Load_Char(fontFace, cp, 0{FT_LOAD_RENDER});
  if err <> 0 then Exit(-1);

  slot := FT_Get_GlyphSlot(fontFace);
  if slot = nil then Exit(-1);
//Exit; ////////////////////////////////////////////////////////////////////////////
  bmp := slot^.bitmap;
}
  Result := GlyphCacheFind(cp);
  if Result >= 0 then
  begin
    pixW := GlyphCache[Result].width;
    pixH := GlyphCache[Result].height;
    left := GlyphCache[Result].left;
    top := GlyphCache[Result].top;
    advance := GlyphCache[Result].advance;
    Exit;
  end;

  // load glyph and render it directly (use FT_LOAD_RENDER)
  err := FT_Load_Char(fontFace, cp, FT_LOAD_RENDER);
  if err <> 0 then Halt{Exit}(-1);

  // obtain glyph slot from face structure
  faceRec := PFT_FaceRec(fontFace);
  if faceRec = nil then Exit(-1);
  slot := faceRec^.glyph;
  if slot = nil then Exit(-1);

  bmp := slot^.bitmap;






  pixW := bmp.width;
  pixH := bmp.rows;
  left := slot^.bitmap_left;
  top := slot^.bitmap_top;
  // advance is in 26.6 fixed, so divide by 64
  advance := slot^.advance.x div 64;

  if (pixW <= 0) or (pixH <= 0) then
  begin
    // empty glyph (e.g., space)
    cacheEntry.codepoint := cp;
    cacheEntry.width := 0;
    cacheEntry.height := 0;
    cacheEntry.left := left;
    cacheEntry.top := top;
    cacheEntry.advance := advance;
    cacheEntry.pixels := nil;
    Result := GlyphCacheAdd(cacheEntry);
    Exit;
  end;

  // Prepare ARGB buffer (premultiplied alpha)
  cacheEntry.codepoint := cp;
  cacheEntry.width := pixW;
  cacheEntry.height := pixH;
  cacheEntry.left := left;
  cacheEntry.top := top;
  cacheEntry.advance := advance;
  cacheEntry.pixels := AllocARGB(pixW, pixH);

  // bmp.pitch may be positive; each row has bmp.width bytes representing coverage (gray)
  rowBytes := NativeUint(bmp.pitch);
  for y := 0 to pixH - 1 do
  begin
    bmpRow := PByte(NativeUInt(bmp.buffer) + NativeUInt(y) * NativeUInt(bmp.pitch));
    // dest row start
    dst := PByte(NativeUInt(cacheEntry.pixels) + NativeUint(y) * NativeUint(pixW) * 4);
    for x := 0 to pixW - 1 do
    begin
      // coverage 0..255
      if x < bmp.width then
        cov := bmpRow[x];
      // store as premultiplied ARGB (we'll multiply by color when drawing)
      // For now keep alpha = cov, and set RGB = 255 (white); actual color applied during composition
      dst[0] := 255;            // B
      dst[1] := 255;            // G
      dst[2] := 255;            // R
      dst[3] := cov;            // A
      Inc(NativeUInt(dst), 4);
    end;
  end;

  Result := GlyphCacheAdd(cacheEntry);
end;

{ Composite an ARGB glyph buffer into a target ARGB buffer with given color (premultiplied alpha) }
procedure CompositeGlyphIntoARGB(dest: Pointer; destW, destH, destStride: Integer;
  src: Pointer; srcW, srcH: Integer; dstX, dstY: Integer; colorARGB: uint32_t);
var
  sx, sy: Integer;
  srcRow: PByte;
  dstRow: PByte;
  sd, dd: PByte;
  cov, srcA: Integer;
  srcR, srcG, srcB: Integer;
  colA, colR, colG, colB: Integer;
  outR, outG, outB, outA: Integer;
  destPtrOffset: NativeUint;
begin
  // colorARGB: 0xAARRGGBB
  colA := (colorARGB shr 24) and $FF;
  colR := (colorARGB shr 16) and $FF;
  colG := (colorARGB shr 8) and $FF;
  colB := (colorARGB shr 0) and $FF;

  for sy := 0 to srcH - 1 do
  begin
    if (dstY + sy < 0) or (dstY + sy >= destH) then Continue;
    srcRow := PByte(NativeUInt(src) + NativeUInt(sy) * NativeUInt(srcW) * 4);
    dstRow := PByte(NativeUInt(dest) + NativeUInt((dstY + sy) * destStride) + NativeUInt(dstX) * 4);
    for sx := 0 to srcW - 1 do
    begin
      // src premultiplied white with alpha = A
      sd := srcRow + sx*4;
      srcA := sd[3]; // 0..255
      if srcA = 0 then
      begin
        // nothing to do
      end
      else
      begin
        // effective source color = color * alpha/255 (premultiplied)
        srcR := (colR * srcA) div 255;
        srcG := (colG * srcA) div 255;
        srcB := (colB * srcA) div 255;
        // dest pixel
        dd := dstRow + sx*4;
        outA := srcA + (dd[3] * (255 - srcA) div 255);
        outR := srcR + (dd[2] * (255 - srcA) div 255);
        outG := srcG + (dd[1] * (255 - srcA) div 255);
        outB := srcB + (dd[0] * (255 - srcA) div 255);
        dd[3] := Byte(outA);
        dd[2] := Byte(outR);
        dd[1] := Byte(outG);
        dd[0] := Byte(outB);
      end;
    end;
  end;
end;

{ Draw UCS4 text at given coordinates with given color (0xAARRGGBB) }
procedure DrawTextUCS4(const s: ucs4; X, Y: Integer; color: uint32_t);
var
  i, ci: Integer;
  idx: Integer;
  gw, gh, left, top, adv: Integer;
  // create temporary ARGB buffer for entire line
  // We'll composite each glyph into line buffer, then put whole line with xcb_put_image
  // compute width
  totalW, totalH: Integer;
  xcur, baseline: Integer;
  tmpBuf: Pointer;
  tmpStride: Integer;
  // pointers for put_image: xcb_put_image wants size and data pointer; we pass tmpBuf
begin
  if s.FLength = 0 then Exit;

  // compute total width and max height (simple: sum advances)
  totalW := 0;
  totalH := 0;
  for i := 0 to s.FLength - 1 do
  begin
    idx := RasterizeGlyph(s.FData[i], gw, gh, left, top, adv);
    if idx < 0 then Continue;
    Inc(totalW, adv);
    if gh > totalH then totalH := gh;
  end;

  if totalW <= 0 then Exit;

  tmpStride := totalW * 4;
  GetMem(tmpBuf, totalH * tmpStride);
  FillChar(tmpBuf^, totalH * tmpStride, 0);

  // composite glyphs left-to-right
  xcur := 0;
  baseline := 0; // we'll place glyph top relative to baseline; simple baseline: font pixel size
  for i := 0 to s.FLength - 1 do
  begin
    ci := GlyphCacheFind(s.FData[i]);
    if ci < 0 then Continue;
    gw := GlyphCache[ci].width;
    gh := GlyphCache[ci].height;
    left := GlyphCache[ci].left;
    top := GlyphCache[ci].top;
    adv := GlyphCache[ci].advance;
    if gw > 0 then
    begin
      // destination x,y within tmpBuf: xcur + left (left may be negative)
      CompositeGlyphIntoARGB(tmpBuf, totalW, totalH, tmpStride, GlyphCache[ci].pixels, gw, gh, xcur + left, totalH - top, color);
    end;
    Inc(xcur, adv);
  end;

  // Now send tmpBuf to X via xcb_put_image (format ZPixmap, depth 32)
  // xcb_put_image(conn, format, drawable, gc, width, height, dst_x, dst_y, left_pad, depth, size, data)
  xcb_put_image(conn, 2{ZPixmap}, win, gc, totalW, totalH, X, Y - totalH, 0, 32, totalW * totalH * 4, tmpBuf);

  FreeMem(tmpBuf);
end;

{ Utilities: get atom by name (intern_atom + reply) }
function GetAtomByName(name: string): xcb_atom_t;
var
  cookie: xcb_void_cookie_t;
  rep: Pointer;
  err: PPxcb_generic_event_t;
begin
  Result := 0;
  cookie := xcb_intern_atom(conn, 0, Length(name), PChar(name));
  rep := xcb_intern_atom_reply(conn, cookie, err);
  if rep = nil then Exit(0);
  // reply struct: first fields include atom at offset ... but easier: cast to pointer and read atom value at 8 bytes offset maybe non-portable.
  // Safer: the reply structure in xcb has atom as uint32 at offset 8. We'll read it directly.
  Result := Puint32_t(NativeUInt(rep) + 8)^;
  cfree(rep);
end;

{ Thread ticker }
function TickerThread(arg: Pointer): Pointer; cdecl;
var b: array[0..0] of Byte;
begin
  while True do
  begin
    b[0] := 1;
    fpwrite(pipe_fds[1], @b[0], 1);
    sleep(1);
  end;
  Result := nil;
end;

{ Main program }
label _cleanup;
var
  setup: Pxcb_setup_t;
  scr_iter: xcb_screen_iterator_t;
  screen_num: cint = 0;
  scr: Pxcb_screen_t;
  maskvals: array[0..1] of uint32_t;
  attrvals: array[0..0] of uint32_t;
  gcvals: array[0..0] of uint32_t;
  cookie: xcb_void_cookie_t;
  ev: Pxcb_generic_event_t;
  buf: array[0..0] of Byte;
  ftd: cint;
  thr: QWord;
  maxfd, s: cint;
  fdset: TFDSet;
  keycode: cuint8;
  keysym: xcb_keysym_t;
  t: TDateTime;
  timeStr, dateStr: string;
  ucs: ucs4;
  lineY: Integer;
cm: Pxcb_client_message_event_t;
begin
  // create pipe
  if fpPipe(pipe_fds) <> 0 then
  begin
    writeln('pipe() failed: ', fpgeterrno);
    Halt(1);
  end;

  // connect to X
  conn := xcb_connect(nil, @screen_num);
  if conn = nil then
  begin
    writeln('xcb_connect returned nil');
    Halt(1);
  end;
  if xcb_connection_has_error(conn) <> 0 then
  begin
    writeln('Cannot connect to X');
    Halt(1);
  end;

  // get screen
  setup := xcb_get_setup(conn);
  scr_iter := xcb_setup_roots_iterator(setup);
  scr := scr_iter.data;
  screen := scr;
  if screen = nil then
  begin
    writeln('No screen');
    Halt(1);
  end;

  // create window
  win := xcb_generate_id(conn);
  maskvals[0] := screen^.white_pixel;
  maskvals[1] := XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_KEY_PRESS or XCB_EVENT_MASK_BUTTON_PRESS;
  cookie := xcb_create_window(conn, screen^.root_depth, win, screen^.root,
    100, 100, 800, 360, 0,
    XCB_WINDOW_CLASS_INPUT_OUTPUT, screen^.root_visual,
    XCB_CW_BACK_PIXEL or XCB_CW_EVENT_MASK, @maskvals[0]);

  // ensure event mask (some servers)
  attrvals[0] := maskvals[1];
  xcb_change_window_attributes(conn, win, XCB_CW_EVENT_MASK, @attrvals[0]);

  xcb_map_window(conn, win);
  xcb_flush(conn);

  // create GC
  gc := xcb_generate_id(conn);
  gcvals[0] := screen^.black_pixel;
  xcb_create_gc(conn, gc, win, XCB_GC_FOREGROUND, @gcvals[0]);

  // keysyms
  keysyms := xcb_key_symbols_alloc(conn);

  // WM_PROTOCOLS / WM_DELETE_WINDOW
  wm_protocols_atom := GetAtomByName('WM_PROTOCOLS');
  wm_delete_atom := GetAtomByName('WM_DELETE_WINDOW');
  if (wm_protocols_atom <> 0) and (wm_delete_atom <> 0) then
  begin
    xcb_change_property(conn, XCB_PROP_MODE_REPLACE, win, wm_protocols_atom, 4 {XA_ATOM}, 32, 1, @wm_delete_atom);
    xcb_flush(conn);
  end;

  // initialize FreeType
  if FT_Init_FreeType(@fontLib) <> 0 then
  begin
    writeln('FT_Init_FreeType failed');
    Halt(1);
  end;
  if FT_New_Face(fontLib, FontPath, 0, @fontFace) <> 0 then
  begin
    writeln('FT_New_Face failed (is DejaVu installed at that path?)');
    Halt(1);
  end;
  FT_Set_Pixel_Sizes(fontFace, 0, FontPixelSize);

  // init glyph cache
  SetLength(GlyphCache, 16);

  // start ticker thread
//  fpThreadCreate(@TickerThread, nil, thr);
pthread_create(@thr, nil, @TickerThread, nil);

  // main loop
  maxfd := xcb_get_file_descriptor(conn);
  if pipe_fds[0] > maxfd then maxfd := pipe_fds[0];
  Inc(maxfd);

  while True do
  begin
    fpFD_ZERO(fdset);
    fpFD_SET(xcb_get_file_descriptor(conn), fdset);
    fpFD_SET(pipe_fds[0], fdset);

    s := fpSelect(maxfd, @fdset, nil, nil, nil);
    if s < 0 then
    begin
      writeln('select error: ', fpgeterrno);
      Break;
    end;

    // X events
    if boolean(fpFD_ISSET(xcb_get_file_descriptor(conn), fdset)) then
    begin
      while True do
      begin
        ev := xcb_poll_for_event(conn);
        if ev = nil then Break;

        case ev^.response_type of
          XCB_EXPOSE:
            begin
              // full redraw on expose
              t := Now;
              timeStr := FormatDateTime('HH:nn:ss', t);
              dateStr := FormatDateTime('dddd, DD MMMM YYYY', t);
              ucs.Init;
              ucs.FromUTF8(timeStr);
              DrawTextUCS4(ucs, 40, 140, $FF1E5AC8); // blue
              ucs.Clear;
              ucs.Init;
              ucs.FromUTF8(dateStr);
              DrawTextUCS4(ucs, 40, 180, $FFD03A3A); // red
              ucs.Clear;
            end;
          XCB_KEY_PRESS:
            begin
              keycode := Pxcb_key_press_event_t(ev)^.detail;
              keysym := xcb_key_symbols_get_keysym(keysyms, keycode, 0);
              // keysyms for latin letters equal ascii codes for lower/upper; 'q' -> 0x71
              if (keysym = Ord('q')) or (keysym = Ord('Q')) or (keysym = $FF1B {Escape}) then
              begin
                cfree(ev);
                goto _cleanup;
              end;
            end;
          {XCB_CLIENT_MESSAGE}161:
            begin
              // cast and check WM_DELETE_WINDOW
              cm:= Pxcb_client_message_event_t(ev);
              if (cm^.event_type = wm_protocols_atom) and (cm^.data.data32[0] = wm_delete_atom) then
              begin
                cfree(ev);
                goto _cleanup;
              end;
            end;
        end;

        cfree(ev);
      end;
    end;

    // pipe (tick)
    if boolean(fpFD_ISSET(pipe_fds[0], fdset)) then
    begin
      if fpRead(pipe_fds[0], @buf[0], 1) > 0 then
      begin
        // redraw time
        t := Now;
        timeStr := FormatDateTime('HH:nn:ss', t);
        dateStr := FormatDateTime('dddd, DD MMMM YYYY', t);

        ucs.Init;
        ucs.FromUTF8(timeStr);
        // clear area where time will be drawn (simple rectangle)
        // create temp blank 32-bit area and put it
        // We'll simply draw text over existing content for simplicity
        DrawTextUCS4(ucs, 40, 140, $FF1E5AC8);
        ucs.Clear;

        ucs.Init;
        ucs.FromUTF8(dateStr);
        DrawTextUCS4(ucs, 40, 180, $FFD03A3A);
        ucs.Clear;

        xcb_flush(conn);
      end;
    end;
  end;

_cleanup:
  // cleanup
  GlyphCacheClear();
  if fontFace <> nil then FT_Done_Face(fontFace);
  if fontLib <> nil then FT_Done_FreeType(fontLib);
  if keysyms <> nil then xcb_key_symbols_free(keysyms);
  xcb_disconnect(conn);
  writeln('Exited');
end.