unit xcb_more_bindings;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}
{$PACKRECORDS C}

interface

uses
  ctypes, BaseUnix, xcb_const, xcb_bindings, xproto_bindings, SysUtils;

type
  // Atoms
  xcb_atom_t = uint32_t;
  Pxcb_atom_t = ^xcb_atom_t;

  // intern atom cookie / reply
  Pxcb_intern_atom_cookie_t = ^xcb_intern_atom_cookie_t;
  xcb_intern_atom_cookie_t = record
    sequence: cuint;
  end;

  Pxcb_intern_atom_reply_t = ^xcb_intern_atom_reply_t;
  xcb_intern_atom_reply_t = record
    response_type: uint8_t;
    pad0: uint8_t;
    sequence: uint16_t;
    length: uint32_t;
    atom: xcb_atom_t;
  end;

  // change_property / get_property cookies and replies
  Pxcb_get_property_cookie_t = ^xcb_get_property_cookie_t;
  xcb_get_property_cookie_t = record
    sequence: cuint;
  end;

  Pxcb_get_property_reply_t = ^xcb_get_property_reply_t;
  xcb_get_property_reply_t = record
    response_type: uint8_t;
    pad0: uint8_t;
    sequence: uint16_t;
    length: uint32_t;
    type_: uint32_t;
    format: uint8_t;
    pad: array[0..3] of uint8_t;
    value_len: uint32_t;
  end;

  // change_window_attributes
  Pxcb_change_window_attributes_value_list_t = ^xcb_change_window_attributes_value_list_t;
  xcb_change_window_attributes_value_list_t = record
    // placeholder; in practice value_list is raw pointer to sequence of uint32_t
  end;

  // Graphics context
  xcb_gcontext_t = uint32_t;

const
  // Some common atoms
  XA_ATOM = 4; // from X11/Xatom.h historically; kept here for convenience (not used for WM atoms directly)

  // WM atoms we will commonly use
  ATOM_WM_PROTOCOLS = 'WM_PROTOCOLS';
  ATOM_WM_DELETE_WINDOW = 'WM_DELETE_WINDOW';

{
  // property modes
  XCB_PROP_MODE_REPLACE = 0;
  XCB_PROP_MODE_PREPEND = 1;
  XCB_PROP_MODE_APPEND = 2;

  // event mask shorthands (some overlap with xproto_bindings)
  XCB_EVENT_MASK_EXPOSURE = 1 shl 15;
  XCB_EVENT_MASK_BUTTON_PRESS = 1 shl 2;
  XCB_EVENT_MASK_BUTTON_RELEASE = 1 shl 3;
  XCB_EVENT_MASK_KEY_PRESS = 1 shl 0;
  XCB_EVENT_MASK_KEY_RELEASE = 1 shl 1;

  // send_event event masks (for SendEvent)
  XCB_SEND_EVENT_DESTINATION_WINDOW = 0;
}

{ External C functions from libxcb (new declarations) }

function xcb_intern_atom(c: Pxcb_connection_t; only_if_exists: uint8_t;
  name_len: uint16_t; name: PChar): xcb_intern_atom_cookie_t; cdecl; external libxcb name 'xcb_intern_atom';
function xcb_intern_atom_reply(c: Pxcb_connection_t; cookie: xcb_intern_atom_cookie_t;
  e: PPxcb_generic_error_t): Pxcb_intern_atom_reply_t; cdecl; external libxcb name 'xcb_intern_atom_reply';

function xcb_change_property(conn: Pxcb_connection_t; mode: uint8_t; window: xcb_window_t;
  _property: xcb_atom_t; type_: xcb_atom_t; format: uint8_t; data_len: cuint;
  data: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_change_property';
function xcb_change_property_checked(conn: Pxcb_connection_t; mode: uint8_t;
  window: xcb_window_t; _property: xcb_atom_t; type_: xcb_atom_t; format: uint8_t;
  data_len: cuint; data: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_change_property_checked';

function xcb_get_property(conn: Pxcb_connection_t; _delete: uint8_t; window: xcb_window_t;
  _property: xcb_atom_t; type_: xcb_atom_t; long_offset: uint32_t; long_length: uint32_t): xcb_get_property_cookie_t; cdecl; external libxcb name 'xcb_get_property';
function xcb_get_property_reply(conn: Pxcb_connection_t; cookie: xcb_get_property_cookie_t;
  e: PPxcb_generic_error_t): Pxcb_get_property_reply_t; cdecl; external libxcb name 'xcb_get_property_reply';
function xcb_get_property_value(reply: Pxcb_get_property_reply_t): Pointer; cdecl; external libxcb name 'xcb_get_property_value';
function xcb_get_property_value_length(reply: Pxcb_get_property_reply_t): csize_t; cdecl; external libxcb name 'xcb_get_property_value_length';

function xcb_change_window_attributes(conn: Pxcb_connection_t; window: xcb_window_t;
  value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_change_window_attributes';
function xcb_change_window_attributes_checked(conn: Pxcb_connection_t; window: xcb_window_t;
  value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_change_window_attributes_checked';

function xcb_create_gc(conn: Pxcb_connection_t; cid: xcb_gcontext_t; drawable: cuint32;
  value_mask: uint32_t; value_list: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_create_gc';

function xcb_send_event(conn: Pxcb_connection_t; propagate: uint8_t; destination: xcb_window_t;
  event_mask: uint32_t; event: Pointer): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_send_event';
function xcb_set_input_focus(conn: Pxcb_connection_t; revert_to: uint8_t; focus: xcb_window_t;
  time: uint32_t): xcb_void_cookie_t; cdecl; external libxcb name 'xcb_set_input_focus';

{ Utility wrappers implemented in Pascal (convenience) }

function GetFirstScreen(conn: Pxcb_connection_t): Pxcb_screen_t;
/// Получает atom по имени. Возвращает 0 при ошибке.
function GetAtom(conn: Pxcb_connection_t; const aName: AnsiString): xcb_atom_t;
/// Установить WM_DELETE_WINDOW для окна (регистрация в WM_PROTOCOLS)
function SetWMProtocols_DeleteWindow(conn: Pxcb_connection_t; win: xcb_window_t): Boolean;
/// Проверка и освобождение generic errors (удобная)
procedure FreeXcbError(err: PPxcb_generic_error_t);

implementation

{ GetFirstScreen }
function GetFirstScreen(conn: Pxcb_connection_t): Pxcb_screen_t;
var
  setup: Pxcb_setup_t;
  iter: xcb_screen_iterator_t;
begin
  if conn = nil then exit(nil);
  setup := xcb_get_setup(conn);
  if setup = nil then exit(nil);
  iter := xcb_setup_roots_iterator(setup);
  // xcb_setup_roots_iterator возвращает итератор, data указывает на первый screen
  Result := iter.data;
end;

{ GetAtom: intern_atom + reply }
function GetAtom(conn: Pxcb_connection_t; const aName: AnsiString): xcb_atom_t;
var
  cookie: xcb_intern_atom_cookie_t;
  reply: Pxcb_intern_atom_reply_t;
  err: PPxcb_generic_error_t;
  pName: PChar;
begin
  Result := 0;
  if conn = nil then Exit;
  pName := PChar(aName);
  cookie := xcb_intern_atom(conn, 0, Length(aName), pName);
  reply := xcb_intern_atom_reply(conn, cookie, @err);
  if reply = nil then
  begin
    // optionally log error
    if err <> nil then
    begin
      // free error
      FreeXcbError(err);
    end;
    Exit(0);
  end;
  Result := reply^.atom;
  cfree(reply); // free reply memory allocated by xcb
end;

{ SetWMProtocols_DeleteWindow }
function SetWMProtocols_DeleteWindow(conn: Pxcb_connection_t; win: xcb_window_t): Boolean;
var
  wm_protocols: xcb_atom_t;
  wm_delete: xcb_atom_t;
  atoms: array[0..0] of xcb_atom_t;
  ck: xcb_void_cookie_t;
  err: PPxcb_generic_error_t;
begin
  Result := False;
  if conn = nil then Exit;
  wm_protocols := GetAtom(conn, ATOM_WM_PROTOCOLS);
  wm_delete := GetAtom(conn, ATOM_WM_DELETE_WINDOW);
  if (wm_protocols = 0) or (wm_delete = 0) then Exit;
  atoms[0] := wm_delete;
  // Replace property WM_PROTOCOLS on the window
  ck := xcb_change_property_checked(conn, XCB_PROP_MODE_REPLACE, win, wm_protocols, XA_ATOM, 32, 1, @atoms[0]);
  err := xcb_request_check(conn, ck);
  if err <> nil then
  begin
    FreeXcbError(err);
    Result := False
  end
  else
    Result := True;
  xcb_flush(conn);
end;

{ FreeXcbError: free generic error pointer }
procedure FreeXcbError(err: PPxcb_generic_error_t);
begin
  if err <> nil then
  begin
    // the pointer err points to Pxcb_generic_error_t; free the memory allocated by libxcb
    cfree(err);
  end;
end;

end.