program generate_xcb_bindings;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}
uses
  Classes, SysUtils, DOM, XMLRead, StrUtils, Math,DateUtils;

const
  INDENT = '  ';

type
  TStringListHelper = class helper for TStringList
    procedure AddIndent(const s: string; level: Integer = 0);
  end;

  TTypeRegistry = class
  private
    FRegisteredTypes: TStringList;
    FBaseNames: TStringList;
  public
    constructor Create;
    destructor Destroy; override;
    function IsRegistered(const typeName: string): Boolean;
    function FindType(const typeName: string): string;
    function RegisterType(const typeName: string; const kind: string): string;
  end;

var
  GlobalTypeRegistry: TTypeRegistry;

function NodeIndex(N: TDOMNode): Integer;
var p: TDOMNode;
begin
  Result := 0;
  p := N.PreviousSibling;
  while Assigned(p) do
  begin
    Inc(Result);
    p := p.PreviousSibling;
  end;
end;

procedure TStringListHelper.AddIndent(const s: string; level: Integer = 0);
var
  i: Integer;
  pref: string;
begin
  pref := StringOfChar(' ', level * Length(INDENT));
  Add(pref + s);
end;

{ TTypeRegistry }

{ TTypeRegistry }
constructor TTypeRegistry.Create;
begin
  inherited Create;
  FRegisteredTypes := TStringList.Create;
  FRegisteredTypes.Sorted := True;
  FRegisteredTypes.Duplicates := dupError;
  FBaseNames := TStringList.Create; // New: store base names for lookup
  FBaseNames.Sorted := True;
  FBaseNames.Duplicates := dupIgnore;
end;

destructor TTypeRegistry.Destroy;
begin
  FRegisteredTypes.Free;
  FBaseNames.Free;
  inherited Destroy;
end;

{ TTypeRegistry }
function TTypeRegistry.RegisterType(const typeName: string; const kind: string): string;
var
  baseName: string;
begin
  baseName := 'T' + typeName;
  
  // For structs, always add suffix to avoid conflicts
  if kind = 'struct' then
    baseName := baseName + '_Struct'
  else if kind = 'union' then
    baseName := baseName + '_Union'
  else if kind = 'event' then
    baseName := baseName + '_Event'
  else if kind = 'error' then
    baseName := baseName + '_Error';
  
  // Use simple names without additional suffixes
  if FRegisteredTypes.IndexOf(baseName) < 0 then
  begin
    FRegisteredTypes.Add(baseName);
  end;
  
  Result := baseName;
end;

function TTypeRegistry.FindType(const typeName: string): string;
var
  baseName: string;
begin
  baseName := 'T' + typeName;
  
  if FRegisteredTypes.IndexOf(baseName) >= 0 then
    Result := baseName
  else
    Result := '';
end;

function TTypeRegistry.IsRegistered(const typeName: string): Boolean;
begin
  Result := FindType(typeName) <> '';
end;

{ --- Evaluate simple expressions --- }
function EvaluateSimpleExpression(const expr: string): string;
var
  parts: TStringArray;
  i, val, shift: Integer;
begin
  Result := Trim(expr);
  
  if Result = '' then
  begin
    Result := '0';
    Exit;
  end;
  
  // Handle hex values: "0x" -> "$"
  if Pos('0x', Result) = 1 then
  begin
    Result := '$' + Copy(Result, 3, MaxInt);
    Exit;
  end;
  
  // Handle binary shifts: "1<<n" -> calculate value
  if Pos('1<<', Result) = 1 then
  begin
    val := 1;
    shift := StrToIntDef(Copy(Result, 4, MaxInt), 0);
    Result := IntToStr(val shl shift);
    Exit;
  end;
  
  // Handle simple additions: "X + Y"
  parts := Result.Split(['+']);
  if Length(parts) = 2 then
  begin
    val := StrToIntDef(Trim(parts[0]), 0) + StrToIntDef(Trim(parts[1]), 0);
    Result := IntToStr(val);
    Exit;
  end;
  
  // Handle simple multiplications: "X * Y"  
  parts := Result.Split(['*']);
  if Length(parts) = 2 then
  begin
    val := StrToIntDef(Trim(parts[0]), 0) * StrToIntDef(Trim(parts[1]), 0);
    Result := IntToStr(val);
    Exit;
  end;
end;

{ --- Normalize type names by replacing colons with underscores --- }
function NormalizeTypeName(const typeName: string): string;
begin
  Result := StringReplace(typeName, ':', '_', [rfReplaceAll]);
end;

{ --- Utility: map XCB types to ctypes --- }


{ --- Emit safe identifier from xml name --- }
function SafeIdent(const s: string): string;
var
  r: string;
  i: Integer;
  keywords: TStringList;
begin
  if s = '' then
  begin
    Result := 'unnamed';
    Exit;
  end;
  
  r := s;
  // replace non-alnum with underscore
  for i := 1 to Length(r) do
    if not (r[i] in ['0'..'9','A'..'Z','a'..'z','_']) then
      r[i] := '_';
  
  // cannot start with digit
  if (Length(r) > 0) and (r[1] in ['0'..'9']) then
    r := 'N' + r;
    
  // Check for Pascal keywords and prefix with underscore
  keywords := TStringList.Create;
  try
    keywords.CommaText := 'record,type,class,property,function,procedure,begin,end,if,then,else,while,do,for,repeat,until,case,of,var,const,array,set,file,program,unit,uses,interface,implementation,initialization,finalization,label,inline,object,constructor,destructor,inherited,packed,absolute,external,forward,mod,div,not,or,and,xor,shl,shr,asm';
    
    if keywords.IndexOf(LowerCase(r)) >= 0 then
      r := '_' + r;
  finally
    keywords.Free;
  end;
  
  Result := r;
end;

{ --- Write unit header --- }
procedure EmitUnitHeader(sl: TStringList; const unitName: string);
begin
  sl.Add('unit ' + unitName + ';');
  sl.Add('');
  sl.Add('{$mode objfpc}{$H+}');
  sl.Add('');
  sl.Add('interface');
  sl.Add('');
  sl.Add('uses');
  sl.AddIndent('ctypes, xcb;', 1);
  sl.Add('');
end;

{ --- Write unit footer --- }
procedure EmitUnitFooter(sl: TStringList);
begin
  sl.Add('');
  sl.Add('implementation');
  sl.Add('');
  sl.Add('end.');
end;

{ --- Parse enums --- }
procedure ProcessEnums(node: TDOMNode; sl: TStringList);
var
  child, enumNode, itemNode, bitNode: TDOMNode;
  enumName: string;
  j, k: Integer;  // Добавлена переменная k
  valName, valValue: string;
  attr: TDOMNode;
  processedEnums: TStringList;
  bitPos: Integer;
begin
  processedEnums := TStringList.Create;
  try
    processedEnums.Sorted := True;
    processedEnums.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if (child.NodeName = 'enum') then
      begin
        enumNode := child;
        attr := enumNode.Attributes.GetNamedItem('name');
        if attr<>nil then
          enumName := attr.NodeValue
        else
          enumName := 'unnamed_enum';

        // Skip if already processed (avoid duplicates)
        if processedEnums.IndexOf(enumName) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedEnums.Add(enumName);

        sl.AddIndent('// enum ' + enumName, 1);

        for j := 0 to enumNode.ChildNodes.Count - 1 do
        begin
          itemNode := enumNode.ChildNodes[j];
          if itemNode.NodeName = 'item' then
          begin
            valName := itemNode.Attributes.GetNamedItem('name').NodeValue;

            // First check for explicit value attribute
            attr := itemNode.Attributes.GetNamedItem('value');
            if attr<>nil then
            begin
              // Use explicit value if present
              valValue := attr.NodeValue;
            end
            else
            begin
              // Check for child <bit> element
              bitNode := nil;
              for k := 0 to itemNode.ChildNodes.Count - 1 do
              begin
                if itemNode.ChildNodes[k].NodeName = 'bit' then
                begin
                  bitNode := itemNode.ChildNodes[k];
                  Break;
                end;
              end;
              
              if bitNode <> nil then
              begin
                // Convert bit position to value: 1 << bit
                bitPos := StrToInt(Trim(bitNode.TextContent));
                valValue := IntToStr(1 shl bitPos);
              end
              else
              begin
                // Fallback: try to get from child text content
                if itemNode.TextContent <> '' then
                  valValue := Trim(itemNode.TextContent)
                else
                  valValue := '0';
              end;
            end;

            // Handle expressions and hex values
            valValue := EvaluateSimpleExpression(valValue);

            sl.AddIndent(Format('const %s = %s;', [SafeIdent(enumName + '_' + valName), valValue]), 1);
          end;
        end;
        sl.Add('');
      end;
      child := child.NextSibling;
    end;
  finally
    processedEnums.Free;
  end;
end;

{ --- Utility: map XCB types to ctypes --- }
function MapXCBTypeToPascal(const xtype: string): string;
var
  x: string;
  foundType: string;
begin
  x := Trim(xtype);
  // Normalize type names first
  x := NormalizeTypeName(x);
  
  // First check if this type is already registered
  foundType := GlobalTypeRegistry.FindType(x);
  if foundType <> '' then
  begin
    Result := foundType;
    Exit;
  end;
  
  // Normalize common XCB types
  if x = '' then Exit('Pointer');
  if SameText(x, 'CARD8') then Exit('cuint8');
  if SameText(x, 'CARD16') then Exit('cuint16');
  if SameText(x, 'CARD32') then Exit('cuint32');
  if SameText(x, 'CARD64') then Exit('cuint64');
  if SameText(x, 'INT8') then Exit('cint8');
  if SameText(x, 'INT16') then Exit('cint16');
  if SameText(x, 'INT32') then Exit('cint32');
  if SameText(x, 'INT64') then Exit('cint64');
  if SameText(x, 'BYTE') then Exit('cuint8');
  if SameText(x, 'BOOL') then Exit('cuint8');
  if SameText(x, 'void') then Exit('Pointer');
  if SameText(x, 'string') or SameText(x, 'str') or SameText(x, 'STRING') then Exit('PAnsiChar');
  if SameText(x, 'float') then Exit('cfloat');
  if SameText(x, 'double') then Exit('cdouble');
  if SameText(x, 'char') then Exit('cchar');
  if SameText(x, 'Timestamp') or SameText(x, 'TIMESTAMP') then Exit('cuint32');
  if SameText(x, 'KeyCode') or SameText(x, 'KEYCODE') then Exit('cuint8');
  if SameText(x, 'KeySym') or SameText(x, 'KEYSYM') then Exit('cuint32');
  
  // X IDs and core types often map to uint32 on 64-bit ABI for xcb (but we keep cuint32)
  if SameText(x, 'Window') or SameText(x, 'Drawable') or SameText(x, 'Atom') or SameText(x, 'Colormap')
    or SameText(x, 'Cursor') or SameText(x, 'VisualID') or SameText(x, 'Time') 
    or SameText(x, 'Pixmap') or SameText(x, 'Region') or SameText(x, 'Rectangle')
    or SameText(x, 'Fence') or SameText(x, 'Crtc') or SameText(x, 'Transform')
    or SameText(x, 'Barrier') or SameText(x, 'Sync') or SameText(x, 'GLX') 
    or SameText(x, 'Behavior') or SameText(x, 'glx_DRAWABLE') 
    or SameText(x, 'DAMAGE') or SameText(x, 'BackBuffer') or SameText(x, 'BufferAttributes')
    or SameText(x, 'CONTEXT_TAG') or SameText(x, 'FLOAT64') or SameText(x, 'FLOAT32')
    or SameText(x, 'BOOL32') or SameText(x, 'EVENT') or SameText(x, 'MODE')
    or SameText(x, 'OUTPUT') or SameText(x, 'PROVIDER') or SameText(x, 'LEASE')
    or SameText(x, 'CrtcChange') or SameText(x, 'OutputChange') or SameText(x, 'OutputProperty')
    or SameText(x, 'ProviderChange') or SameText(x, 'ProviderProperty') or SameText(x, 'ResourceChange')
    or SameText(x, 'LeaseNotify') or SameText(x, 'NotifyData') or SameText(x, 'Range8')
    or SameText(x, 'Range16') or SameText(x, 'ExtRange') or SameText(x, 'ClientSpec')
    or SameText(x, 'ElementHeader') or SameText(x, 'PICTFORMAT') or SameText(x, 'DIRECTFORMAT')
    or SameText(x, 'FIXED') or SameText(x, 'POINTFIX') or SameText(x, 'LINEFIX')
    or SameText(x, 'COUNTER') or SameText(x, 'sync_INT64') or SameText(x, 'TRIGGER')
    or SameText(x, 'ALARM') or SameText(x, 'ClientIdSpec') or SameText(x, 'ResourceIdSpec')
    or SameText(x, 'ResourceSizeSpec') or SameText(x, 'KIND') or SameText(x, 'SEG')
    or SameText(x, 'DOTCLOCK') or SameText(x, 'EventTypeBase') or SameText(x, 'DeviceId')
    or SameText(x, 'FP3232') or SameText(x, 'DefaultBehavior') or SameText(x, 'LockBehavior')
    or SameText(x, 'RadioGroupBehavior') or SameText(x, 'OverlayBehavior') or SameText(x, 'SASetMods')
    or SameText(x, 'SASetGroup') or SameText(x, 'SASetControls') or SameText(x, 'LedClassSpec')
    or SameText(x, 'IDSpec') or SameText(x, 'SIAction') or SameText(x, 'CommonBehavior')
    or SameText(x, 'PermamentLockBehavior') or SameText(x, 'PermamentRadioGroupBehavior')
    or SameText(x, 'PermamentOverlayBehavior') or SameText(x, 'SANoAction') or SameText(x, 'SALatchMods')
    or SameText(x, 'SALockMods') or SameText(x, 'SASetGroup') or SameText(x, 'SALatchGroup')
    or SameText(x, 'PCONTEXT') or SameText(x, 'SURFACE') or SameText(x, 'PORT')
    or SameText(x, 'ENCODING') or SameText(x, 'Rational') or SameText(x, 'VISUALID')
    or SameText(x, 'ATOM') or SameText(x, 'COLORMAP') or SameText(x, 'KeySymMap')
    or SameText(x, 'TRANSFORM') or SameText(x, 'PICTURE') or SameText(x, 'GLYPHSET')
    or SameText(x, 'GLYPH') or SameText(x, 'SPANFIX') or SameText(x, 'ModifierInfo')
    or SameText(x, 'GroupInfo') or SameText(x, 'DeviceState') or SameText(x, 'SAMovePtr')
    or SameText(x, 'SAPtrBtn') or SameText(x, 'SALockPtrBtn') or SameText(x, 'SASetPtrDflt')
    or SameText(x, 'SAIsoLock') or SameText(x, 'SATerminate') or SameText(x, 'SASwitchScreen')
    or SameText(x, 'SAActionMessage') or SameText(x, 'SARedirectKey') or SameText(x, 'SADeviceBtn')
    or SameText(x, 'SALockDeviceBtn') or SameText(x, 'SADeviceValuator') or SameText(x, 'CHARINFO')
    or SameText(x, 'ModifierInfo') or SameText(x, 'GroupInfo') or SameText(x, 'DeviceState') then 
  begin
    Result := 'cuint32';
    Exit;
  end;
  
  // Fallback: if starts with 'CARD' numeric -> cuint32 default
  if StartsText('CARD', UpperCase(x)) then Exit('cuint32');
  if StartsText('INT', UpperCase(x)) then Exit('cint32');

  // If all else fails, register it as a new type
  Result := GlobalTypeRegistry.RegisterType(x, 'unknown');
end;

{ --- Process typedefs --- }
procedure ProcessTypedefs(node: TDOMNode; sl: TStringList);
var
  child: TDOMNode;
  name, toType: string;
  processedTypes: TStringList;
begin
  processedTypes := TStringList.Create;
  try
    processedTypes.Sorted := True;
    processedTypes.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if child.NodeName = 'typedef' then
      begin
        name := child.Attributes.GetNamedItem('newname').NodeValue;
        
        // Skip if already processed
        if processedTypes.IndexOf(name) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedTypes.Add(name);
        
        if child.Attributes.GetNamedItem('oldname') <> nil then
          toType := child.Attributes.GetNamedItem('oldname').NodeValue
        else if child.TextContent <> '' then
          toType := Trim(child.TextContent)
        else
          toType := '';
          
        // Normalize type names
        toType := NormalizeTypeName(toType);
        name := NormalizeTypeName(name);
        
        sl.AddIndent('// typedef ' + name + ' -> ' + toType, 1);
        if toType <> '' then
          sl.AddIndent(Format('type %s = %s;', [GlobalTypeRegistry.RegisterType(name, 'typedef'), MapXCBTypeToPascal(toType)]), 1)
        else
          sl.AddIndent(Format('type %s = Pointer; // typedef %s (unknown target)', [GlobalTypeRegistry.RegisterType(name, 'typedef'), name]), 1);
        sl.Add('');
      end;
      child := child.NextSibling;
    end;
  finally
    processedTypes.Free;
  end;
end;

{ --- Process structs --- }
{
procedure ProcessStructs(node: TDOMNode; sl: TStringList);
var
  child, structNode, fieldNode: TDOMNode;
  i, padBytes: Integer;
  structName, fname, ftype: string;
  attrNode: TDOMNode;
  processedStructs: TStringList;
begin
  processedStructs := TStringList.Create;
  try
    processedStructs.Sorted := True;
    processedStructs.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if SameText(child.NodeName, 'struct') then
      begin
        structNode := child;
        attrNode := structNode.Attributes.GetNamedItem('name');
        if Assigned(attrNode) then
          structName := attrNode.NodeValue
        else
          structName := 'unnamed_struct';

        // Skip if already processed
        if processedStructs.IndexOf(structName) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedStructs.Add(structName);

        WriteLn('  Processing struct: ', structName);
        
        sl.AddIndent('// struct ' + structName, 1);
        sl.AddIndent('type', 1);
        sl.AddIndent(Format('%s = packed record', [GlobalTypeRegistry.RegisterType(structName, 'struct')]), 2);

        for i := 0 to structNode.ChildNodes.Count - 1 do
        begin
          fieldNode := structNode.ChildNodes[i];

          if SameText(fieldNode.NodeName, 'field') then
          begin
            // field name
            if Assigned(fieldNode.Attributes.GetNamedItem('name')) then
              fname := SafeIdent(fieldNode.Attributes.GetNamedItem('name').NodeValue)
            else
              fname := Format('field_%d', [NodeIndex(fieldNode)]);

            // field type (may be absent for list/str)
            if Assigned(fieldNode.Attributes.GetNamedItem('type')) then
            begin
              ftype := fieldNode.Attributes.GetNamedItem('type').NodeValue;
              // Normalize type name
              ftype := NormalizeTypeName(ftype);
            end
            else
              ftype := '';

            // Special case: field named "type" conflicts with Pascal keyword
            if fname = 'type' then
              fname := '_type';

            // handle common varlen/list/str as Pointer with comment
            if (ftype = '') or SameText(ftype, 'list') or SameText(ftype, 'str') or SameText(ftype, 'STRING') then
            begin
              sl.AddIndent(Format('%s: Pointer; // original: %s (node index %d)', [fname, Trim(fieldNode.TextContent), NodeIndex(fieldNode)]), 3);
            end
            else
            begin
              sl.AddIndent(Format('%s: %s;', [fname, MapXCBTypeToPascal(ftype)]), 3);
            end;
          end
          else if SameText(fieldNode.NodeName, 'pad') then
          begin
            // pad element: try to read bytes attr
            if Assigned(fieldNode.Attributes.GetNamedItem('bytes')) then
            begin
              try
                padBytes := StrToInt(fieldNode.Attributes.GetNamedItem('bytes').NodeValue);
                if padBytes > 0 then
                  sl.AddIndent(Format('pad_%d: array[0..%d-1] of cuint8; // padding', [NodeIndex(fieldNode), padBytes]), 3)
                else
                  sl.AddIndent(Format('pad_%d: cuint8; // pad (0 bytes)', [NodeIndex(fieldNode)]), 3);
              except
                on E: Exception do
                  sl.AddIndent(Format('pad_%d: cuint8; // pad (invalid bytes: %s)', [NodeIndex(fieldNode), fieldNode.Attributes.GetNamedItem('bytes').NodeValue]), 3);
              end;
            end
            else
              sl.AddIndent(Format('pad_%d: cuint8; // pad (unspecified bytes)', [NodeIndex(fieldNode)]), 3);
          end
          else if SameText(fieldNode.NodeName, 'doc') then
          begin
            // documentation node — ignore
          end
          else if SameText(fieldNode.NodeName, 'list') then
          begin
            // list field - treat as pointer for now
            sl.AddIndent(Format('list_%d: Pointer; // list field', [NodeIndex(fieldNode)]), 3);
          end
          else if SameText(fieldNode.NodeName, 'switch') then
          begin
            // switch field - complex case, skip for now
            sl.AddIndent(Format('switch_%d: Pointer; // switch field - manual handling required', [NodeIndex(fieldNode)]), 3);
          end
          else
          begin
            // unknown/special nodes - skip complex processing
            sl.AddIndent(Format('// %s (node index %d) - skipped for performance', [fieldNode.NodeName, NodeIndex(fieldNode)]), 3);
          end;
        end;

        sl.AddIndent('end;', 2);
        sl.Add('');
      end;

      child := child.NextSibling;
    end;
  finally
    processedStructs.Free;
  end;
end;
}

{ --- Process events/errors (as typedef of record) --- }
procedure ProcessEventsErrors(node: TDOMNode; sl: TStringList; kind: string);
var
  child, enode, fieldNode: TDOMNode;
  name, fname, ftype: string;
  i: Integer;
  processedItems: TStringList;
begin
  processedItems := TStringList.Create;
  try
    processedItems.Sorted := True;
    processedItems.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if SameText(child.NodeName, kind) then
      begin
        enode := child;
        name := enode.Attributes.GetNamedItem('name').NodeValue;
        
        // Skip if already processed
        if processedItems.IndexOf(name) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedItems.Add(name);
        
        sl.AddIndent('// ' + kind + ' ' + name, 1);
        sl.AddIndent('type', 1);
        sl.AddIndent(Format('%s = packed record', [GlobalTypeRegistry.RegisterType(name, kind)]), 2);
        for i := 0 to enode.ChildNodes.Count - 1 do
        begin
          fieldNode := enode.ChildNodes[i];
          if fieldNode.NodeName = 'field' then
          begin
            fname := SafeIdent(fieldNode.Attributes.GetNamedItem('name').NodeValue);
            
            // Special case: field named "type"
            if fname = 'type' then
              fname := '_type';
              
            if fieldNode.Attributes.GetNamedItem('type') <> nil then
            begin
              ftype := fieldNode.Attributes.GetNamedItem('type').NodeValue;
              // Normalize type name
              ftype := NormalizeTypeName(ftype);
            end
            else
              ftype := '';
              
            if (ftype = '') or SameText(ftype, 'str') or SameText(ftype, 'list') then
              sl.AddIndent(Format('%s: Pointer; // original: %s', [fname, fieldNode.TextContent]), 3)
            else
              sl.AddIndent(Format('%s: %s;', [fname, MapXCBTypeToPascal(ftype)]), 3);
          end;
        end;
        sl.AddIndent('end;', 2);
        sl.Add('');
      end;
      child := child.NextSibling;
    end;
  finally
    processedItems.Free;
  end;
end;

{ --- Process requests (generate opcode const and reply record if present) --- }
procedure ProcessRequests(node: TDOMNode; sl: TStringList);
var
  child, rnode, replyNode, fieldNode: TDOMNode;
  name, opcode, ftype: string;
  i, j: Integer;
  processedRequests: TStringList;
begin
  processedRequests := TStringList.Create;
  try
    processedRequests.Sorted := True;
    processedRequests.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if child.NodeName = 'request' then
      begin
        rnode := child;
        name := rnode.Attributes.GetNamedItem('name').NodeValue;
        
        // Skip if already processed
        if processedRequests.IndexOf(name) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedRequests.Add(name);
        
        opcode := '0'; // may be in attribute opcode
        if rnode.Attributes.GetNamedItem('opcode') <> nil then
          opcode := rnode.Attributes.GetNamedItem('opcode').NodeValue;
        sl.AddIndent('// request ' + name, 1);
        sl.AddIndent(Format('const XCB_%s = %s; // opcode', [UpperCase(name), opcode]), 1);
        // reply
        replyNode := nil;
        for i := 0 to rnode.ChildNodes.Count - 1 do
        begin
          if rnode.ChildNodes[i].NodeName = 'reply' then
          begin
            replyNode := rnode.ChildNodes[i];
            Break;
          end;
        end;
        if Assigned(replyNode) then
        begin
          sl.AddIndent('type', 1);
          sl.AddIndent(Format('%sReply = packed record', [GlobalTypeRegistry.RegisterType(name, 'request')]), 2);
          for j := 0 to replyNode.ChildNodes.Count - 1 do
          begin
            fieldNode := replyNode.ChildNodes[j];
            if fieldNode.NodeName = 'field' then
            begin
              if fieldNode.Attributes.GetNamedItem('type') <> nil then
              begin
                ftype := fieldNode.Attributes.GetNamedItem('type').NodeValue;
                ftype := NormalizeTypeName(ftype);
                sl.AddIndent(Format('%s: %s;', [SafeIdent(fieldNode.Attributes.GetNamedItem('name').NodeValue),
                  MapXCBTypeToPascal(ftype)]), 3)
              end
              else
                sl.AddIndent(Format('%s: Pointer; // varlen/unknown', [SafeIdent(fieldNode.Attributes.GetNamedItem('name').NodeValue)]), 3);
            end;
          end;
          sl.AddIndent('end;', 2);
          sl.Add('');
        end;
      end;
      child := child.NextSibling;
    end;
  finally
    processedRequests.Free;
  end;
end;

{ --- Process xid types --- }
procedure ProcessXidTypes(node: TDOMNode; sl: TStringList);
var
  child: TDOMNode;
  name: string;
  processedXids: TStringList;
begin
  processedXids := TStringList.Create;
  try
    processedXids.Sorted := True;
    processedXids.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if child.NodeName = 'xidtype' then
      begin
        name := child.Attributes.GetNamedItem('name').NodeValue;
        
        // Skip if already processed
        if processedXids.IndexOf(name) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedXids.Add(name);
        
        sl.AddIndent('// xidtype ' + name, 1);
        // Register and use simple names for xidtypes
        sl.AddIndent(Format('type %s = cuint32;', [GlobalTypeRegistry.RegisterType(name, 'xidtype')]), 1);
        sl.Add('');
      end;
      child := child.NextSibling;
    end;
  finally
    processedXids.Free;
  end;
end;

{ --- Process structs --- }
procedure ProcessStructs(node: TDOMNode; sl: TStringList);
var
  child, structNode, fieldNode: TDOMNode;
  i, padBytes: Integer;
  structName, fname, ftype, finalTypeName: string;
  attrNode: TDOMNode;
  processedStructs: TStringList;
begin
  processedStructs := TStringList.Create;
  try
    processedStructs.Sorted := True;
    processedStructs.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if SameText(child.NodeName, 'struct') then
      begin
        structNode := child;
        attrNode := structNode.Attributes.GetNamedItem('name');
        if Assigned(attrNode) then
          structName := attrNode.NodeValue
        else
          structName := 'unnamed_struct';

        // Skip if already processed
        if processedStructs.IndexOf(structName) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedStructs.Add(structName);

        // Check if this name conflicts with already registered types
        if GlobalTypeRegistry.IsRegistered(structName) then
        begin
          // Use suffix for structs that conflict with xidtypes
          finalTypeName := GlobalTypeRegistry.RegisterType(structName + '_Struct', 'struct');
        end
        else
        begin
          finalTypeName := GlobalTypeRegistry.RegisterType(structName, 'struct');
        end;

        sl.AddIndent('// struct ' + structName, 1);
        sl.AddIndent('type', 1);
        sl.AddIndent(Format('%s = packed record', [finalTypeName]), 2);

        for i := 0 to structNode.ChildNodes.Count - 1 do
        begin
          fieldNode := structNode.ChildNodes[i];

          if SameText(fieldNode.NodeName, 'field') then
          begin
            // field name
            if Assigned(fieldNode.Attributes.GetNamedItem('name')) then
              fname := SafeIdent(fieldNode.Attributes.GetNamedItem('name').NodeValue)
            else
              fname := Format('field_%d', [NodeIndex(fieldNode)]);

            // field type (may be absent for list/str)
            if Assigned(fieldNode.Attributes.GetNamedItem('type')) then
            begin
              ftype := fieldNode.Attributes.GetNamedItem('type').NodeValue;
              // Normalize type name
              ftype := NormalizeTypeName(ftype);
            end
            else
              ftype := '';

            // Special case: field named "type" conflicts with Pascal keyword
            if fname = 'type' then
              fname := '_type';

            // handle common varlen/list/str as Pointer with comment
            if (ftype = '') or SameText(ftype, 'list') or SameText(ftype, 'str') or SameText(ftype, 'STRING') then
            begin
              sl.AddIndent(Format('%s: Pointer; // original: %s', [fname, Trim(fieldNode.TextContent)]), 3);
            end
            else
            begin
              sl.AddIndent(Format('%s: %s;', [fname, MapXCBTypeToPascal(ftype)]), 3);
            end;
          end
          else if SameText(fieldNode.NodeName, 'pad') then
          begin
            // pad element: try to read bytes attr
            if Assigned(fieldNode.Attributes.GetNamedItem('bytes')) then
            begin
              try
                padBytes := StrToInt(fieldNode.Attributes.GetNamedItem('bytes').NodeValue);
                if padBytes > 0 then
                  sl.AddIndent(Format('pad_%d: array[0..%d-1] of cuint8; // padding', [NodeIndex(fieldNode), padBytes]), 3)
                else
                  sl.AddIndent(Format('pad_%d: cuint8; // pad (0 bytes)', [NodeIndex(fieldNode)]), 3);
              except
                on E: Exception do
                  sl.AddIndent(Format('pad_%d: cuint8; // pad (invalid bytes: %s)', [NodeIndex(fieldNode), fieldNode.Attributes.GetNamedItem('bytes').NodeValue]), 3);
              end;
            end
            else
              sl.AddIndent(Format('pad_%d: cuint8; // pad (unspecified bytes)', [NodeIndex(fieldNode)]), 3);
          end
          else if SameText(fieldNode.NodeName, 'doc') then
          begin
            // documentation node — ignore
          end
          else if SameText(fieldNode.NodeName, 'list') then
          begin
            // list field - treat as pointer for now
            sl.AddIndent(Format('list_%d: Pointer; // list field', [NodeIndex(fieldNode)]), 3);
          end
          else if SameText(fieldNode.NodeName, 'switch') then
          begin
            // switch field - complex case, skip for now
            sl.AddIndent(Format('switch_%d: Pointer; // switch field - manual handling required', [NodeIndex(fieldNode)]), 3);
          end
          else
          begin
            // unknown/special nodes - skip complex processing
            sl.AddIndent(Format('// %s (node index %d) - skipped', [fieldNode.NodeName, NodeIndex(fieldNode)]), 3);
          end;
        end;

        sl.AddIndent('end;', 2);
        sl.Add('');
      end;

      child := child.NextSibling;
    end;
  finally
    processedStructs.Free;
  end;
end;

{ --- Process constants (value) --- }
procedure ProcessValues(node: TDOMNode; sl: TStringList);
var
  child, vnode: TDOMNode;
  name, val: string;
  processedValues: TStringList;
begin
  processedValues := TStringList.Create;
  try
    processedValues.Sorted := True;
    processedValues.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if child.NodeName = 'value' then
      begin
        vnode := child;
        name := SafeIdent(vnode.Attributes.GetNamedItem('name').NodeValue);
        
        // Skip if already processed
        if processedValues.IndexOf(name) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedValues.Add(name);
        
        if vnode.Attributes.GetNamedItem('value') <> nil then
          val := vnode.Attributes.GetNamedItem('value').NodeValue
        else
          val := vnode.TextContent;
          
        // Handle expressions and hex values
        val := EvaluateSimpleExpression(val);
          
        sl.AddIndent(Format('const %s = %s;', [name, val]), 1);
      end;
      child := child.NextSibling;
    end;
    if (sl.Count > 0) and (sl[sl.Count-1] <> '') then sl.Add('');
  finally
    processedValues.Free;
  end;
end;

{ --- Process unions --- }
procedure ProcessUnions(node: TDOMNode; sl: TStringList);
var
  child, unionNode, fieldNode: TDOMNode;
  i: Integer;
  unionName, fname, ftype, listSize: string;
  processedUnions: TStringList;
  hasFields: Boolean;
begin
  processedUnions := TStringList.Create;
  try
    processedUnions.Sorted := True;
    processedUnions.Duplicates := dupIgnore;
    
    child := node.FirstChild;
    while Assigned(child) do
    begin
      if SameText(child.NodeName, 'union') then
      begin
        unionNode := child;
        unionName := unionNode.Attributes.GetNamedItem('name').NodeValue;
        
        // Skip if already processed
        if processedUnions.IndexOf(unionName) >= 0 then
        begin
          child := child.NextSibling;
          Continue;
        end;
        processedUnions.Add(unionName);

        sl.AddIndent('// union ' + unionName, 1);
        sl.AddIndent('type', 1);
        sl.AddIndent(Format('%s = packed record', [GlobalTypeRegistry.RegisterType(unionName, 'union')]), 2);
        
        // Check if union has any fields
        hasFields := False;
        for i := 0 to unionNode.ChildNodes.Count - 1 do
        begin
          if SameText(unionNode.ChildNodes[i].NodeName, 'field') or 
             SameText(unionNode.ChildNodes[i].NodeName, 'list') then
          begin
            hasFields := True;
            Break;
          end;
        end;
        
        if hasFields then
        begin
          sl.AddIndent('case Integer of', 3);

          for i := 0 to unionNode.ChildNodes.Count - 1 do
          begin
            fieldNode := unionNode.ChildNodes[i];
            if SameText(fieldNode.NodeName, 'field') or SameText(fieldNode.NodeName, 'list') then
            begin
              if Assigned(fieldNode.Attributes.GetNamedItem('name')) then
                fname := SafeIdent(fieldNode.Attributes.GetNamedItem('name').NodeValue)
              else
                fname := Format('field_%d', [i]);
                
              if fieldNode.Attributes.GetNamedItem('type') <> nil then
              begin
                ftype := fieldNode.Attributes.GetNamedItem('type').NodeValue;
                ftype := NormalizeTypeName(ftype);
              end
              else
                ftype := 'Pointer';

              // Handle list with size
              if SameText(fieldNode.NodeName, 'list') then
              begin
                // Get list size from value child node
                listSize := '1';
                if Assigned(fieldNode.FirstChild) and (fieldNode.FirstChild.NodeName = 'value') then
                  listSize := fieldNode.FirstChild.TextContent
                else if Assigned(fieldNode.Attributes.GetNamedItem('value')) then
                  listSize := fieldNode.Attributes.GetNamedItem('value').NodeValue;
                  
                sl.AddIndent(Format('%d: (%s: array[0..%s-1] of %s);', 
                  [i, fname, listSize, MapXCBTypeToPascal(ftype)]), 4);
              end
              else
              begin
                sl.AddIndent(Format('%d: (%s: %s);', [i, fname, MapXCBTypeToPascal(ftype)]), 4);
              end;
            end;
          end;
        end
        else
        begin
          // Empty union - add dummy field
          sl.AddIndent('dummy: cuint8; // empty union', 3);
        end;

        sl.AddIndent('end;', 2);
        sl.Add('');
      end;
      child := child.NextSibling;
    end;
  finally
    processedUnions.Free;
  end;
end;

{ --- Check if XML file is valid (not XSD schema) --- }
function IsValidXCBXML(const xmlPath: string): Boolean;
var
  doc: TXMLDocument;
  root: TDOMNode;
begin
  Result := False;
  if not FileExists(xmlPath) then Exit;
  
  try
    ReadXMLFile(doc, xmlPath);
    try
      root := doc.DocumentElement;
      if not Assigned(root) then Exit;
      
      // Check if it's a real XCB protocol file, not XSD schema
      if (root.NodeName = 'xcb') or (Pos('protocol', LowerCase(xmlPath)) > 0) then
        Result := True;
    finally
      doc.Free;
    end;
  except
    on E: Exception do
      Result := False;
  end;
end;

{ --- Main: parse a single xml file and emit .pas --- }
procedure ProcessXMLFile(const xmlPath: string; const outDir: string);
var
  doc: TXMLDocument;
  root: TDOMNode;
  unitName, outFile: string;
  sl: TStringList;
  xmlFilename: string;
  startTime: TDateTime;
begin
  // Skip XSD schema files and invalid XML
  if not IsValidXCBXML(xmlPath) then
  begin
    WriteLn('Skipping invalid or XSD file: ', xmlPath);
    Exit;
  end;
  
  startTime := Now;
  WriteLn('Parsing ', xmlPath);
  
  // Skip if processing takes too long
  if MinutesBetween(Now, startTime) > 2 then // 2 minutes timeout
  begin
    WriteLn('Timeout processing ', xmlPath);
    Exit;
  end;
  
  try
    ReadXMLFile(doc, xmlPath);
  except
    on E: Exception do
    begin
      WriteLn('Error reading XML ', xmlPath, ': ', E.Message);
      Exit;
    end;
  end;
  
  root := doc.DocumentElement;
  if not Assigned(root) then
  begin
    WriteLn('No document element in ', xmlPath);
    doc.Free;
    Exit;
  end;
  
  if root.Attributes.GetNamedItem('name') <> nil then
    unitName := root.Attributes.GetNamedItem('name').NodeValue
  else
    unitName := ChangeFileExt(ExtractFileName(xmlPath), '');
    
  unitName := SafeIdent(LowerCase(unitName)); // use lower-case safe name
  
  // Special handling for record.xml -> _record.pas
  if unitName = 'record' then
    unitName := '_record';
    
  outFile := IncludeTrailingPathDelimiter(outDir) + unitName + '.pas';

  sl := TStringList.Create;
  try
    EmitUnitHeader(sl, unitName);

    // Add short auto-generated comment
    sl.AddIndent('// Auto-generated from ' + ExtractFileName(xmlPath), 1);
    sl.AddIndent('// Pure 1:1 C binding skeleton for FreePascal (packed records, consts).', 1);
    sl.Add('');

    // process common nodes in logical order with timeout checks
    if MinutesBetween(Now, startTime) < 2 then ProcessXidTypes(root, sl);
    if MinutesBetween(Now, startTime) < 2 then ProcessTypedefs(root, sl);
    if MinutesBetween(Now, startTime) < 2 then ProcessEnums(root, sl);
    if MinutesBetween(Now, startTime) < 2 then ProcessValues(root, sl);
    if MinutesBetween(Now, startTime) < 2 then ProcessStructs(root, sl);
    if MinutesBetween(Now, startTime) < 2 then ProcessUnions(root, sl);
    if MinutesBetween(Now, startTime) < 2 then ProcessEventsErrors(root, sl, 'event');
    if MinutesBetween(Now, startTime) < 2 then ProcessEventsErrors(root, sl, 'error');
    if MinutesBetween(Now, startTime) < 2 then ProcessRequests(root, sl);

    EmitUnitFooter(sl);

    sl.SaveToFile(outFile);
    WriteLn('Wrote ', outFile, ' in ', MilliSecondsBetween(Now, startTime), ' ms');
  finally
    sl.Free;
    doc.Free;
  end;
end;

{ --- Main program entry --- }
var
  i: Integer;
  srcDir, outDir: string;
  sr: TSearchRec;
  found: Integer;
begin
  GlobalTypeRegistry := TTypeRegistry.Create;
  try
    if ParamCount < 1 then
    begin
      WriteLn('Usage: generate_xcb_bindings <path-to-xcb-proto-xml-dir> [outdir]');
      Halt(1);
    end;
    srcDir := ParamStr(1);
    if ParamCount >= 2 then outDir := ParamStr(2) else outDir := srcDir;

    if not DirectoryExists(srcDir) then
    begin
      WriteLn('Directory not found: ', srcDir);
      Halt(1);
    end;

    // Create output directory if it doesn't exist
    if not DirectoryExists(outDir) then
      CreateDir(outDir);

    // iterate xml files
    found := 0;
    if FindFirst(IncludeTrailingPathDelimiter(srcDir) + '*.xml', faAnyFile, sr) = 0 then
    begin
      repeat
        ProcessXMLFile(IncludeTrailingPathDelimiter(srcDir) + sr.Name, outDir);
        Inc(found);
      until FindNext(sr) <> 0;
      FindClose(sr);
    end;

    if found = 0 then
      WriteLn('No .xml files found in ', srcDir)
    else
      WriteLn('Processed ', found, ' files.');
  finally
    GlobalTypeRegistry.Free;
  end;
end.