program test_advanced_window;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  ctypes, xcb_bindings, BaseUnix;

var
  conn: Pxcb_connection_t;
  setup: Pxcb_setup_t;
  screen_iter: Txcb_screen_iterator_t;
  screen: Pxcb_screen_t;
  window: cuint32;
  cookie: Txcb_void_cookie_t;
  error: Pxcb_generic_error_t;
  event: Pxcb_generic_event_t;
  running: Boolean = True;

procedure HandleEvent(event: Pxcb_generic_event_t);
begin
  case event^.response_type of
    // Key press/release events
    2, 3: // KeyPress/KeyRelease
      begin
        WriteLn('Keyboard event: ', event^.response_type);
        if event^.response_type = 2 then
          running := False; // Exit on key press
      end;
    
    // Button press/release events  
    4, 5: // ButtonPress/ButtonRelease
      begin
        WriteLn('Mouse button event: ', event^.response_type);
      end;
    
    // Expose event (window needs redraw)
    12: // Expose
      begin
        WriteLn('Expose event - window needs redraw');
      end;
    
    // Configure notify (window resized/moved)
    22: // ConfigureNotify
      begin
        WriteLn('Configure event - window geometry changed');
      end;
    
    // Client message (usually close request)
    33: // ClientMessage
      begin
        WriteLn('Client message - closing window');
        running := False;
      end;
    
    else
      WriteLn('Unknown event: ', event^.response_type);
  end;
end;

begin
  WriteLn('Testing advanced XCB window with event handling...');

  // Connect to X server
  conn := xcb_connect(nil, nil);
  if xcb_connection_has_error(conn) <> 0 then
  begin
    WriteLn('Error: Cannot connect to X server');
    Halt(1);
  end;

  // Get first screen
  setup := xcb_get_setup(conn);
  screen_iter := xcb_setup_roots_iterator(setup);
  screen := screen_iter.data;
  
  WriteLn('Screen: ', screen^.width_in_pixels, 'x', screen^.height_in_pixels);

  // Create window
  window := xcb_generate_id(conn);
  
  WriteLn('Creating window...');
  cookie := xcb_create_window(
    conn,
    XCB_COPY_FROM_PARENT,
    window,
    screen^.root,
    100, 100,  // x, y
    400, 300,  // width, height
    10,        // border width
    XCB_WINDOW_CLASS_INPUT_OUTPUT,
    screen^.root_visual,
    0, nil
  );

  // Set window title
  WriteLn('Setting window title...');
  cookie := xcb_change_property(
    conn,
    XCB_PROP_MODE_REPLACE,
    window,
    1, // WM_NAME
    1, // STRING
    8, // 8-bit format
    13, // length of "XCB Test Window"
    PChar('XCB Test Window')
  );

  // Subscribe to events
  WriteLn('Subscribing to events...');
  cookie := xcb_change_window_attributes_checked(
    conn,
    window,
    XCB_EVENT_MASK_BASIC, // Subscribe to basic events
    nil
  );

  error := xcb_request_check(conn, cookie);
  if error <> nil then
  begin
    WriteLn('Error: Cannot set event mask');
    Freemem(error);
    xcb_destroy_window(conn, window);
    xcb_disconnect(conn);
    Halt(1);
  end;

  // Map window
  WriteLn('Mapping window...');
  cookie := xcb_map_window(conn, window);
  
  // Flush to ensure requests are sent
  xcb_flush(conn);
  WriteLn('Window created and mapped. Press any key in window to exit...');

  // Event loop
  while running do
  begin
    event := xcb_poll_for_event(conn);
    if event <> nil then
    begin
      HandleEvent(event);
      free(event); // Use standard C free instead of xcb_free_event
    end
    else
    begin
      // No events, sleep a bit to avoid busy waiting
      fpSleep(1);
    end;
  end;

  // Cleanup
  WriteLn('Cleaning up...');
  xcb_destroy_window(conn, window);
  xcb_disconnect(conn);
  
  WriteLn('Test completed successfully!');
end.