program test_xcb;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

uses
  xcb, xproto, xcb_bindings, ctypes,SysUtils;

var
  conn: Pxcb_connection_t;
  screen: Pxcb_screen_t;
  window: Txcb_window_t;
  cookie: Txcb_void_cookie_t;
  error: Pxcb_generic_error_t;
  geom_cookie: Txcb_get_geometry_cookie_t;
  geom_reply: Pxcb_get_geometry_reply_t;

begin
  WriteLn('Testing XCB bindings...');

  // Connect to X server
  conn := xcb_connect(nil, nil);
  if xcb_connection_has_error(conn) <> 0 then
  begin
    WriteLn('Error: Cannot connect to X server');
    Halt(1);
  end;
  WriteLn('Connected to X server');

  // Get first screen
  screen := xcb_setup_roots_iterator(xcb_get_setup(conn)).data;
  if screen = nil then
  begin
    WriteLn('Error: Cannot get screen');
    xcb_disconnect(conn);
    Halt(1);
  end;
  WriteLn('Got screen: ', screen^.width, 'x', screen^.height);

  // Create window
  window := xcb_generate_id(conn);
  
  cookie := xcb_create_window_checked(
    conn,
    XCB_COPY_FROM_PARENT,
    window,
    screen^.root,
    100, 100,  // x, y
    400, 300,  // width, height
    10,        // border width
    XCB_WINDOW_CLASS_INPUT_OUTPUT,
    screen^.root_visual,
    0, nil
  );

  error := xcb_request_check(conn, cookie);
  if error <> nil then
  begin
    WriteLn('Error: Cannot create window');
    FreeMem(error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  WriteLn('Window created: ', window);

  // Map window
  cookie := xcb_map_window_checked(conn, window);
  error := xcb_request_check(conn, cookie);
  if error <> nil then
  begin
    WriteLn('Error: Cannot map window');
    FreeMem(error);
    xcb_disconnect(conn);
    Halt(1);
  end;
  WriteLn('Window mapped');

  // Flush to ensure requests are sent
  xcb_flush(conn);
  WriteLn('Requests flushed');

  // Test getting window geometry
  geom_cookie := xcb_get_geometry(conn, window);
  geom_reply := xcb_get_geometry_reply(conn, geom_cookie, nil);
  
  if geom_reply <> nil then
  begin
    WriteLn('Window geometry:');
    WriteLn('  Position: ', geom_reply^.x, 'x', geom_reply^.y);
    WriteLn('  Size: ', geom_reply^.width, 'x', geom_reply^.height);
    WriteLn('  Border: ', geom_reply^.border_width);
    FreeMem(geom_reply);
  end
  else
  begin
    WriteLn('Error: Cannot get window geometry');
  end;

  // Wait a bit to see the window
  WriteLn('Waiting 3 seconds...');
  Sleep(3000);

  // Cleanup
  xcb_destroy_window(conn, window);
  xcb_disconnect(conn);
  
  WriteLn('Test completed successfully!');
end.