unit xcb_bindings;

{
    XCB Bindings.
    For GNU/Linux.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026  Artyomov Alexander
    http://self-made-free.ru/
    Used https://chat.deepseek.com/, https://chatgpt.com/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.
                              
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

interface

uses
  ctypes;

const
  libxcb = 'libxcb.so';

type
  // Basic types from xcb.h
  Pxcb_connection_t = ^Txcb_connection_t;
  Txcb_connection_t = record end;

  // Keycode type (should match xproto.pas)
  xcb_keycode_t = cuint8;

  Pxcb_screen_t = ^Txcb_screen_t;
  Txcb_screen_t = record
    // Basic fields we need for testing
    root: cuint32;
    default_colormap: cuint32;
    white_pixel: cuint32;
    black_pixel: cuint32;
    current_input_masks: cuint32;
    width_in_pixels: cuint16;
    height_in_pixels: cuint16;
    width_in_millimeters: cuint16;
    height_in_millimeters: cuint16;
    min_installed_maps: cuint16;
    max_installed_maps: cuint16;
    root_visual: cuint32;
    backing_stores: cint8;
    save_unders: cuint8;
    root_depth: cuint8;
    allowed_depths_len: cuint8;
  end;

  Pxcb_setup_t = ^Txcb_setup_t;
  Txcb_setup_t = record
    status: cuint8;
    pad0: cuint8;
    protocol_major_version: cuint16;
    protocol_minor_version: cuint16;
    length: cuint16;
    release_number: cuint32;
    resource_id_base: cuint32;
    resource_id_mask: cuint32;
    motion_buffer_size: cuint32;
    vendor_len: cuint16;
    maximum_request_length: cuint16;
    roots_len: cuint8;
    pixmap_formats_len: cuint8;
    image_byte_order: cuint8;
    bitmap_format_bit_order: cuint8;
    bitmap_format_scanline_unit: cuint8;
    bitmap_format_scanline_pad: cuint8;
    min_keycode: xcb_keycode_t;
    max_keycode: xcb_keycode_t;
    pad1: array[0..3] of cuint8;
  end;

  Pxcb_screen_iterator_t = ^Txcb_screen_iterator_t;
  Txcb_screen_iterator_t = record
    data: Pxcb_screen_t;
    rem: cint;
    index: cint;
  end;

  Txcb_void_cookie_t = record
    sequence: cuint;
  end;

  Pxcb_generic_error_t = ^Txcb_generic_error_t;
  Txcb_generic_error_t = record
    response_type: cuint8;
    error_code: cuint8;
    sequence: cuint16;
    resource_id: cuint32;
    minor_code: cuint16;
    major_code: cuint8;
    pad0: cuint8;
    pad: array[0..4] of cuint32;
    full_sequence: cuint32;
  end;

  PPxcb_generic_error_t = ^Pxcb_generic_error_t;

  // Event types
  Pxcb_generic_event_t = ^Txcb_generic_event_t;
  Txcb_generic_event_t = record
    response_type: cuint8;
    pad0: cuint8;
    sequence: cuint16;
    pad: array[0..6] of cuint32;
    full_sequence: cuint32;
  end;

  // Geometry functions
  Pxcb_get_geometry_cookie_t = ^Txcb_get_geometry_cookie_t;
  Txcb_get_geometry_cookie_t = record
    sequence: cuint;
  end;

  Pxcb_get_geometry_reply_t = ^Txcb_get_geometry_reply_t;
  Txcb_get_geometry_reply_t = record
    response_type: cuint8;
    depth: cuint8;
    sequence: cuint16;
    length: cuint32;
    root: cuint32;
    x: cint16;
    y: cint16;
    width: cuint16;
    height: cuint16;
    border_width: cuint16;
    pad0: array[0..1] of cuint8;
  end;

  // ===== BASIC CONNECTION FUNCTIONS =====
  function xcb_connect(display: PChar; screenp: Pcint): Pxcb_connection_t; cdecl; external libxcb;
  function xcb_connection_has_error(conn: Pxcb_connection_t): cint; cdecl; external libxcb;
  procedure xcb_disconnect(conn: Pxcb_connection_t); cdecl; external libxcb;
  procedure xcb_flush(conn: Pxcb_connection_t); cdecl; external libxcb;

  function xcb_get_setup(conn: Pxcb_connection_t): Pxcb_setup_t; cdecl; external libxcb;
  function xcb_setup_roots_iterator(setup: Pxcb_setup_t): Txcb_screen_iterator_t; cdecl; external libxcb;
  procedure xcb_screen_next(iter: Pxcb_screen_iterator_t); cdecl; external libxcb;

  function xcb_generate_id(conn: Pxcb_connection_t): cuint32; cdecl; external libxcb;
  function xcb_request_check(conn: Pxcb_connection_t; cookie: Txcb_void_cookie_t): Pxcb_generic_error_t; cdecl; external libxcb;

  // ===== WINDOW MANAGEMENT =====
  function xcb_create_window_checked(
    conn: Pxcb_connection_t;
    depth: cuint8;
    wid: cuint32;
    parent: cuint32;
    x: cint16;
    y: cint16;
    width: cuint16;
    height: cuint16;
    border_width: cuint16;
    _class: cuint16;
    visual: cuint32;
    value_mask: cuint32;
    value_list: Pointer
  ): Txcb_void_cookie_t; cdecl; external libxcb;

  function xcb_create_window(
    conn: Pxcb_connection_t;
    depth: cuint8;
    wid: cuint32;
    parent: cuint32;
    x: cint16;
    y: cint16;
    width: cuint16;
    height: cuint16;
    border_width: cuint16;
    _class: cuint16;
    visual: cuint32;
    value_mask: cuint32;
    value_list: Pointer
  ): Txcb_void_cookie_t; cdecl; external libxcb;

  function xcb_map_window_checked(conn: Pxcb_connection_t; window: cuint32): Txcb_void_cookie_t; cdecl; external libxcb;
  function xcb_map_window(conn: Pxcb_connection_t; window: cuint32): Txcb_void_cookie_t; cdecl; external libxcb;
  function xcb_unmap_window_checked(conn: Pxcb_connection_t; window: cuint32): Txcb_void_cookie_t; cdecl; external libxcb;
  function xcb_destroy_window_checked(conn: Pxcb_connection_t; window: cuint32): Txcb_void_cookie_t; cdecl; external libxcb;
  function xcb_destroy_window(conn: Pxcb_connection_t; window: cuint32): Txcb_void_cookie_t; cdecl; external libxcb;

  // ===== EVENT HANDLING =====
  function xcb_poll_for_event(conn: Pxcb_connection_t): Pxcb_generic_event_t; cdecl; external libxcb;
  function xcb_poll_for_queued_event(conn: Pxcb_connection_t): Pxcb_generic_event_t; cdecl; external libxcb;
  function xcb_wait_for_event(conn: Pxcb_connection_t): Pxcb_generic_event_t; cdecl; external libxcb;
  procedure free(event: Pointer); cdecl; external 'c' name 'free'; // Use standard C free

  function xcb_change_window_attributes_checked(
    conn: Pxcb_connection_t;
    window: cuint32;
    value_mask: cuint32;
    value_list: Pointer
  ): Txcb_void_cookie_t; cdecl; external libxcb;

  // ===== PROPERTIES AND TITLES =====
  function xcb_change_property_checked(
    conn: Pxcb_connection_t;
    mode: cuint8;
    window: cuint32;
    _property: cuint32;
    _type: cuint32;
    format: cuint8;
    data_len: cuint32;
    data: Pointer
  ): Txcb_void_cookie_t; cdecl; external libxcb;

  function xcb_change_property(
    conn: Pxcb_connection_t;
    mode: cuint8;
    window: cuint32;
    _property: cuint32;
    _type: cuint32;
    format: cuint8;
    data_len: cuint32;
    data: Pointer
  ): Txcb_void_cookie_t; cdecl; external libxcb;

  // ===== GEOMETRY FUNCTIONS =====
  function xcb_get_geometry(conn: Pxcb_connection_t; drawable: cuint32): Txcb_get_geometry_cookie_t; cdecl; external libxcb;
  function xcb_get_geometry_reply(conn: Pxcb_connection_t; cookie: Txcb_get_geometry_cookie_t; error: PPxcb_generic_error_t): Pxcb_get_geometry_reply_t; cdecl; external libxcb;

// Constants from xproto.h
const
  XCB_COPY_FROM_PARENT = 0;
  XCB_WINDOW_CLASS_INPUT_OUTPUT = 1;
  
  // Property modes
  XCB_PROP_MODE_REPLACE = 0;
  XCB_PROP_MODE_PREPEND = 1;
  XCB_PROP_MODE_APPEND = 2;
  
  // Event masks (from our generated xproto.pas)
  XCB_EVENT_MASK_NO_EVENT = 0;
  XCB_EVENT_MASK_KEY_PRESS = 1;
  XCB_EVENT_MASK_KEY_RELEASE = 2;
  XCB_EVENT_MASK_BUTTON_PRESS = 4;
  XCB_EVENT_MASK_BUTTON_RELEASE = 8;
  XCB_EVENT_MASK_ENTER_WINDOW = 16;
  XCB_EVENT_MASK_LEAVE_WINDOW = 32;
  XCB_EVENT_MASK_POINTER_MOTION = 64;
  XCB_EVENT_MASK_POINTER_MOTION_HINT = 128;
  XCB_EVENT_MASK_BUTTON1_MOTION = 256;
  XCB_EVENT_MASK_BUTTON2_MOTION = 512;
  XCB_EVENT_MASK_BUTTON3_MOTION = 1024;
  XCB_EVENT_MASK_BUTTON4_MOTION = 2048;
  XCB_EVENT_MASK_BUTTON5_MOTION = 4096;
  XCB_EVENT_MASK_BUTTON_MOTION = 8192;
  XCB_EVENT_MASK_KEYMAP_STATE = 16384;
  XCB_EVENT_MASK_EXPOSURE = 32768;
  XCB_EVENT_MASK_VISIBILITY_CHANGE = 65536;
  XCB_EVENT_MASK_STRUCTURE_NOTIFY = 131072;
  XCB_EVENT_MASK_RESIZE_REDIRECT = 262144;
  XCB_EVENT_MASK_SUBSTRUCTURE_NOTIFY = 524288;
  XCB_EVENT_MASK_SUBSTRUCTURE_REDIRECT = 1048576;
  XCB_EVENT_MASK_FOCUS_CHANGE = 2097152;
  XCB_EVENT_MASK_PROPERTY_CHANGE = 4194304;
  XCB_EVENT_MASK_COLOR_MAP_CHANGE = 8388608;
  XCB_EVENT_MASK_OWNER_GRAB_BUTTON = 16777216;

  // Common event masks
  XCB_EVENT_MASK_BASIC = XCB_EVENT_MASK_KEY_PRESS or XCB_EVENT_MASK_KEY_RELEASE or
                        XCB_EVENT_MASK_BUTTON_PRESS or XCB_EVENT_MASK_BUTTON_RELEASE or
                        XCB_EVENT_MASK_EXPOSURE or XCB_EVENT_MASK_STRUCTURE_NOTIFY;

implementation

end.